/* Interface for a vector-like data structure */

/*
 * The Vector stores void pointers to user-allocated data.
 *
 * User-supplied data pointers may be NULL, in which case they are
 * treated the same as nonexistent data.
 *
 * The Vector may contain holes, that is, elements not occupied by
 * user data (or by NULL user data; no distinction is made between the
 * two) that are followed by elements containing (non-NULL) user data.
 */
#ifndef _Vector_h_
#define _Vector_h_

typedef struct Vector_t Vector; /* opaque type */


/*--------------------------------------------------------------------------*
 * Create a new, empty Vector
 * After use, the Vector must be freed using vecDelete().
 *--------------------------------------------------------------------------*
 * Return: a pointer to the new vector
 *         If this pointer is NULL, an error has occurred.
 *--------------------------------------------------------------------------*/
Vector* vecCreate(void);


/*--------------------------------------------------------------------------*
 * Delete a Vector.
 *--------------------------------------------------------------------------*
 * Parameter: a pointer to a vector created using vecCreate()
 *--------------------------------------------------------------------------*/
void vecDelete(Vector* vector);


/*--------------------------------------------------------------------------*
 * Query the size of a vector
 *--------------------------------------------------------------------------*
 * Parameter: pointer to a Vector
 *
 * Return: 1 + the maximal index currently occupied by (non-NULL) user data, or
 *         0 if there are none
 *--------------------------------------------------------------------------*/
int vecSize(const Vector* vector);


/*--------------------------------------------------------------------------*
 * Access the data stored at a given index in the Vector
 *--------------------------------------------------------------------------*
 * Parameter: 
 *   pointer to a Vector
 *   index of desired element
 *
 * Return: pointer to the data stored at the requested index
 *         If there are no data associated with this index, the
 *         returned pointer is NULL.
 *--------------------------------------------------------------------------*/
void* vecGet(const Vector* vector, int index);


/*--------------------------------------------------------------------------*
 * Set the data at a given index in the Vector
 * User data may or may not already exist at the given index.
 * The index may be (much) larger than the current value of vecSize(),
 * in which case the Vector will contain empty cells after this call.
 *--------------------------------------------------------------------------*
 * Parameter: 
 *   pointer to a Vector
 *   index of desired element
 *   pointer to the data to be associated with this index
 *
 * Return: 0 if ok; nonzero if error
 *         If the return value is nonzero, no change has been made to
 *         the Vector.
 *--------------------------------------------------------------------------*/
int vecSet(Vector* vector, int index, void* data);


/*--------------------------------------------------------------------------*
 * Insert data at a given index in the Vector. Any existing data at
 * this index, and at all following indices, are moved up by one index
 * in the Vector.
 * This is the reverse operation of vecRemove().
 *--------------------------------------------------------------------------*
 * Parameter:
 *   pointer to a Vector
 *   index of desired element
 *   pointer to the data to be inserted at this index
 *
 * Return: 0 if ok; nonzero if error
 *         If the return value is nonzero, no change has been made to
 *         the Vector.
 *--------------------------------------------------------------------------*/
int vecInsert(Vector* vector, int index, void* data);


/*--------------------------------------------------------------------------*
 * Remove the data from a given index in the Vector. Any existing data at
 * this index, and at all following indices, are moved down by one index.
 * This is the reverse operation of vecInsert().
 *--------------------------------------------------------------------------*
 * Parameter:
 *   pointer to a Vector
 *   index of the element to be removed
 *
 * Return: the data associated with this index as defined for vecGet()
 *--------------------------------------------------------------------------*/
void* vecRemove(Vector* vector, int index);


/*--------------------------------------------------------------------------*
 * Reverse the Vector such that the data pointer formerly stored at
 * index 0 is located at index vecSize() - 1, and vice versa, and
 * equivalently for all elements in-between.
 *--------------------------------------------------------------------------*
 * Parameter: pointer to a Vector
 *--------------------------------------------------------------------------*/
void vecReverse(Vector* vector);

#endif /* _Vector_h_ */
