#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define GNUPLT_PARAM_FILE "diffusion.param.gnuplt"
#define OUTPUT_DATA_FILE "diffusion.dat"

#define NPOINTS    512
#define NITERS     100000
#define NOUT       1000
#define XLEN       2.0
#define DIFF_COEF  1.0

void write_step(FILE* datfile, 
                int total_size, 
                double* u, double t, double xlen, double dx) {
  double x;

  for(int i = 1; i <= total_size; i++) {
    x = dx * ((double)i - 0.5) - 0.5 * XLEN;
    fprintf(datfile, " %24.14lf %24.14lf %24.14lf\n", t, x, u[i]);
  }

  fprintf(datfile, "\n");
}

int main(int argc, char *argv[]) {
  double dx, dt, dtdx2, x, t;
  
  FILE* datfile = NULL;
  FILE* paramfile = NULL;

  double *uold, *unew;

  dx = XLEN / NPOINTS;
  dt = 0.5 * dx * dx / DIFF_COEF;
  dtdx2 = dt / (dx * dx);

  t = 0.0;
      
  uold = (double*)malloc(sizeof(double)*(NPOINTS+2));
  unew = (double*)malloc(sizeof(double)*(NPOINTS+2));

  // Initial conditions
  for(int i = 1; i <= NPOINTS; i++) {
    x = dx * ((double)i - 0.5) - 0.5 * XLEN;
    uold[i] = 0.5 * cos(2.0 * M_PI * x / XLEN) + 0.5;
  }

  uold[0] = 0.0;
  unew[0] = 0.0;
      
  uold[NPOINTS+1] = 0.0;
  unew[NPOINTS+1] = 0.0;

  datfile = fopen(OUTPUT_DATA_FILE, "w");
  paramfile = fopen(GNUPLT_PARAM_FILE, "w");

  fprintf(paramfile, "npoints = %d\n", NPOINTS);
  fprintf(paramfile, "niters = %d\n", NITERS);
  fprintf(paramfile, "nout = %d\n", NOUT);
  fprintf(paramfile, "xlen = %lf\n", XLEN);
  fprintf(paramfile, "dt = %lf\n", dt);
  fclose(paramfile);

  write_step(datfile, NPOINTS, uold, t, XLEN, dx);

  for (int iter = 1; iter <= NITERS; iter++) {
    for (int i = 1; i <= NPOINTS; i++) {
      unew[i] = uold[i] + DIFF_COEF * dtdx2
                        * (uold[i+1] - 2.0 * uold[i] + uold[i-1]);
    }

    if (iter % NOUT == 0)
      write_step(datfile, NPOINTS, uold, t, XLEN, dx);
        
    t += dt;

    double *tmpptr = uold;
    uold = unew;
    unew = tmpptr;
  }

  fclose(datfile);
  
  free(uold);
  free(unew);

  return 0;
}