/**
 * This simple program demonstrates the use of the fprinf and fscanf standard 
 * library functions by writing and reading back a 2D array in a text file.
 * 
 * Compile with: gcc -o text_file text_file.c
 * Run with: ./text_file
 * 
 * In order to read or write a file, we first need to open it. This is done with
 * the fopen function that has the following signature:
 * 
 *     FILE *fopen(const char *filename, const char *mode);
 * 
 *     filename - name of the file to read or write
 *         mode - character string determining file access mode=
 *                  - "wb": write binary mode
 *                  - "rb": read binary mode
 * 
 * 
 * To write a file we use the fwrite function. The fwrite function has the
 * following signature:
 * 
 *     size_t fwrite(void *buffer, 
 *                   size_t size,
 *                   size_t count,
 *                   FILE *stream);
 *  
 *     buffer - pointer to the array where the elements we write are stored
 *       size - size of each element in bytes
 *      count - the number of elements to write
 *     stream - the stream to write (obtained with fopen)
 * 
 * 
 * To read a file we use the fread function. The fread function has the
 * following signature:
 * 
 *     size_t fread(void *buffer, 
 *                  size_t size,
 *                  size_t count,
 *                  FILE *stream);
 *  
 *     buffer - pointer to the array where the elements we read will be stored
 *       size - size of each element in bytes
 *      count - the number of elements to be read
 *     stream - the stream to read (obtained with fopen)
 *  
 * Once we have finished to write or read a file, we need to close it so that
 * any unwritten buffered data are flushed to the operating system. This is done
 * with the fclose function that has the following signature:
 * 
 *     int fclose(FILE *stream);
 * 
 *     stream - the file stream to close
 */


#include <stdio.h>  // for fprintf, fopen, fread, fwrite
#include <stdlib.h> // for exit, malloc and free
#include <assert.h> // for assert

#define N 8

/**
 * Convenient macro that is used to check that function calls return SUCCESS.
 * This macro will call the process_error function if function call fails.
 */
#define CHECK(expr)                                       \
do {                                                      \
  error_t err;                                            \
  if ((err = (expr)) != SUCCESS)                          \
    process_error(err, __FILE__, __FUNCTION__, __LINE__); \
} while(0)

typedef enum {
  SUCCESS = 0,
  ALLOCATION_FAILED,
  FILE_OPEN_FAILED,
  FILE_READ_FAILED,
  FILE_WRITE_FAILED
} error_t;

static const char* error_desc[] = {
  "Success",
  "Allocation failed",
  "Failed to open file",
  "Failed to read file",
  "Failed to write file"
};

void process_error(error_t err, const char *file, const char *func, int line) {
  printf("**ERROR** %s (in %s() %s:%d)\n", error_desc[err], func, file, line);
  exit(err);
}

error_t read_array(double** array, int* size, const char *filename) {
  int lsize;
  double *larray;
  FILE *fp;

  assert(   array != NULL);
  assert(filename != NULL);

  /** Open the file in read (r) */
  if ((fp = fopen(filename, "r")) == NULL)
    return FILE_OPEN_FAILED;

  /** 
   * Read the data in two stages:
   *  - the first call to the fscanf function reads the size of the array
   *  - in a second stage we do multiple call to the fscanf function to read
   *    the array. One call per value.
   * 
   * At every step we check that the read is a success by checking the return
   * value of fscanf. This return value should be equal to the number of 
   * elements we read.
   */
  if (fscanf(fp, "%d", &lsize) != 1) {
    fclose(fp);
    return FILE_READ_FAILED;
  }

  assert(lsize > 0);

  /** We need to allocate the array so that we have space to read the data */
  if ((larray = (double*)malloc(sizeof(double) * lsize)) == NULL ) {
    fclose(fp);
    return ALLOCATION_FAILED;
  }

  for (int i = 0; i < lsize; i++) {
    if (fscanf(fp, "%lf", &larray[i]) != 1) {
      fclose(fp);
      return FILE_READ_FAILED;
    }
  }

  *array = larray;
  *size  = lsize;
  
  fclose(fp);

  return SUCCESS;
}

int write_array(double *array, int size, const char *filename) {
  assert(   array != NULL);
  assert(filename != NULL);

  /** Open the file in write (w) binary (b) mode */
  FILE *fp;
  if ((fp = fopen(filename, "w")) == NULL)
    return FILE_OPEN_FAILED;

  /** 
   * Write the data in two stages:
   *  - the first call to the fprintf function writes the size of the array
   *  - in a second stage we do multiple call to the fprinf function to write
   *    the array. One call per value.
   * 
   * At every step we check that the write is a success by checking the return
   * value of fprinf. If the return value is negative it has failed.
   */
  assert(size > 0);

  if (fprintf(fp, "%d\n", size) < 0) {
    /** Even if it fails we have to make sure to close the file */
    fclose(fp);
    return FILE_WRITE_FAILED;
  }

  for (int i = 0; i < size; i++) {
    if (fprintf(fp, "%12.6lf", array[i]) < 0) {
      fclose(fp);
      return FILE_WRITE_FAILED;
    }
  }

  /** Close the file after a successful write */
  fclose(fp);

  return SUCCESS;
}

void print_array(double *array, int size) {
  assert(array != NULL);

  for (int i = 0; i < size; i++)
    printf("%6.1lf", array[i]);

  printf("\n");
}

int main(int argc, char* argv[]) {
  double* array_write;
  double* array_read;
  int read_size;

  CHECK(
    (array_write = (double*)malloc(sizeof(double) * N)) == NULL 
      ? ALLOCATION_FAILED 
      : SUCCESS
  );

  /** Fill the array with values: 1.0, 2.0, ... */
  double value = 0.0;
  for (int i = 0; i < N; i++) {
      array_write[i] = ++value;
  }

  printf("Array is:\n");
  print_array(array_write, N);

  printf("Writting array...\n");
  CHECK( write_array(array_write, N, "array.txt") );

  printf("Reading array...\n");
  CHECK( read_array(&array_read, &read_size, "array.txt") );

  printf("Array is:\n");
  print_array(array_read, read_size);

  /** Free memory before exiting */
  free(array_write);
  free(array_read);

  return SUCCESS;
}