#include <algorithm>

#include "average.hh"
#include "grade.hh"
#include "student_info.hh"

using namespace std;

static bool did_all_hws(const Student_info& s) {
    return find(s.homeworks.begin(), s.homeworks.end(), 0)
        == s.homeworks.end();
}

static void write_analysis(
        ostream& out, const string& name,
        double analysis(const vector<Student_info>&),
        const vector<Student_info>& did,
        const vector<Student_info>& didnt)
{
    out << name << ": median(did) = " << analysis(did)
        << ", median(didnt) = " << analysis(didnt) << endl;
}

static double grade_aux(const Student_info& s) {
    try { return grade(s); }
    catch (domain_error) {
        return grade(s.midterm, s.final, 0);
    }
}

static double median_analysis(
        const vector<Student_info>& students)
{
    vector<double> grades;
    transform(students.begin(), students.end(),
              back_inserter(grades), grade_aux);
    return median(grades);
}

static double average_grade(const Student_info& s) {
    return grade(s.midterm, s.final, average(s.homeworks));
}

static double average_analysis(
        const vector<Student_info>& students)
{
    vector<double> grades;
    transform(students.begin(), students.end(),
              back_inserter(grades), average_grade);
    return median(grades);
}

// Median of the nonzero elements of `s.homeworks`, or 0 if none
static double optimistic_median(const Student_info& s) {
    vector<double> nonzero;
    remove_copy(s.homeworks.begin(), s.homeworks.end(),
                back_inserter(nonzero), 0);
    double homework_grade = nonzero.empty() ? 0 : median(nonzero);
    return grade(s.midterm, s.final, homework_grade);
}

static double optimistic_median_analysis(
        const vector<Student_info>& students) {
    vector<double> grades;
    transform(students.begin(), students.end(),
              back_inserter(grades), optimistic_median);
    return median(grades);
}

int main() {
    // Read the student records and partition them
    vector<Student_info> did, didnt;
    Student_info student;
    while (read(cin, student)) {
        if (did_all_hws(student)) did.push_back(student);
        else didnt.push_back(student);
    }
    // Verify that the analyses will show us something
    if (did.empty()) {
        cerr << "No student did all the homeworks!" << endl;
        return 1;
    }
    if (didnt.empty()) {
        cerr << "Every student did all the homeworks!" << endl;
        return 1;
    }
    // Do the analyses
    write_analysis(cout, "median", median_analysis, did, didnt);
    write_analysis(cout, "average", average_analysis, did, didnt);
    write_analysis(cout, "median of homework turned in",
                   optimistic_median_analysis, did, didnt);

    return 0;
}
