#ifndef VEC3_HH
#define VEC3_HH

#include <cstddef>
#include <memory>

template<class T> class Vec {
public:
    typedef T* iterator;
    typedef const T* const_iterator;
    typedef std::size_t size_type;
    typedef T value_type;
    typedef T& reference;
    typedef const T& const_reference;

    Vec() { create(); }
    explicit Vec(size_type n, const T& t = T()) {
        create(n, t);
    }
    Vec(const Vec& v) { create(v.begin(), v.end()); }
    ~Vec() { uncreate(); }

    size_type size() const { return limit - data; }

    T& operator[](size_type i) { return data[i]; }
    const T& operator[](size_type i) const {
        return data[i];
    }

    iterator begin() { return data; }
    const_iterator begin() const { return data; }

    iterator end() { return limit; }
    const_iterator end() const { return limit; }

    Vec& operator=(const Vec&);

private:
    void create();
    void create(size_type n, const T& t);
    void create(const_iterator begin, const_iterator end);
    void uncreate();

    iterator data;  // First element in the `Vec`
    iterator limit; // One past the allocated memory

    std::allocator<T> alloc;
};

template<class T> Vec<T>& Vec<T>::operator=(const Vec& rhs) {
    if (&rhs != this) {
        uncreate();
        create(rhs.begin(), rhs.end());
    }
    return *this;
}

template<class T> void Vec<T>::create() {
    data = limit = nullptr;
}

template<class T> void Vec<T>::create(size_type n, const T& val) {
    data = alloc.allocate(n);
    limit = std::uninitialized_fill_n(data, n, val);
}

template<class T> void Vec<T>::create(const_iterator b, const_iterator e) {
    data = alloc.allocate(e - b);
    limit = std::uninitialized_copy(b, e, data);
}

template <class T> void Vec<T>::uncreate() {
    if (data) {
         // Destroy (in reverse order) the elements that were constructed
        iterator it = limit;
        while (it != data)
            alloc.destroy(--it);

        // Return all the space that was allocated
        alloc.deallocate(data, limit - data);
    }
    // Reset pointers to indicate that the `Vec` is empty again
    data = limit = nullptr;
}

#endif
