#ifndef VEC4_HH
#define VEC4_HH

#include <cstddef>
#include <memory>

template<class T> class Vec {
public:
    typedef T* iterator;
    typedef const T* const_iterator;
    typedef std::size_t size_type;
    typedef T value_type;
    typedef T& reference;
    typedef const T& const_reference;

    Vec() { create(); }
    explicit Vec(size_type n, const T& t = T()) {
        create(n, t);
    }
    Vec(const Vec& v) { create(v.begin(), v.end()); }
    ~Vec() { uncreate(); }

    size_type size() const { return avail - data; }

    T& operator[](size_type i) { return data[i]; }
    const T& operator[](size_type i) const {
        return data[i];
    }

    iterator begin() { return data; }
    const_iterator begin() const { return data; }

    iterator end() { return avail; }
    const_iterator end() const { return avail; }

    void push_back(const T& t) {
        if (avail == limit)
            grow();
        unchecked_append(t);
    }

    Vec& operator=(const Vec&);

private:
    void create();
    void create(size_type n, const T& t);
    void create(const_iterator begin, const_iterator end);
    void uncreate();
    void grow();
    void unchecked_append(const T& t);

    iterator data;  // First element in the `Vec`
    iterator avail;  // One past initialised elements
    iterator limit; // One past the allocated memory

    std::allocator<T> alloc;
};

template<class T> Vec<T>& Vec<T>::operator=(const Vec& rhs) {
    if (&rhs != this) {
        uncreate();
        create(rhs.begin(), rhs.end());
    }
    return *this;
}

template<class T>
void Vec<T>::create() {
    data = limit = avail = nullptr;
}

template<class T>
void Vec<T>::create(size_type n, const T& val) {
    data = alloc.allocate(n);
    limit = avail = std::uninitialized_fill_n(data, n, val);
}

template<class T>
void Vec<T>::create(const_iterator b, const_iterator e) {
    data = alloc.allocate(e - b);
    limit = avail = std::uninitialized_copy(b, e, data);
}

template <class T>
void Vec<T>::uncreate() {
    if (data) {
         // Destroy initalialised elements in reverse order
        iterator it = avail;
        while (it != data)
            alloc.destroy(--it);

        // Return all the space that was allocated
        alloc.deallocate(data, limit - data);
    }
    // Reset pointers to indicate this `Vec` is empty
    data = limit = avail = nullptr;
}

template<class T> void Vec<T>::grow() {
    // When growing, allocate twice as much space as currently in use
    size_type new_size = max(2 * (limit - data), std::ptrdiff_t(1));

    // Allocate new space and copy existing elements to the new space
    iterator new_data = alloc.allocate(new_size);
    iterator new_avail = std::uninitialized_copy(data, avail, new_data);

    // Return the old space
    uncreate();

    // Reset pointers to point to the newly allocated space
    data = new_data;
    avail = new_avail;
    limit = data + new_size;
}

// Assumes `avail` points at allocated, but uninitialized space
template <class T> void Vec<T>::unchecked_append(const T& val) {
    alloc.construct(avail++, val);
}

#endif
