/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009, 2010, 2011 Sébastien PIERARD
 */

#include <iostream>
#include <cassert>

#include "Granu.h"
#include "Matrix.h"

using namespace std ;

static inline int min ( int a , int b ) {
	return ( a < b ) ? a : b ;
}

void Granu :: findLargestIncludedRectangles ( const Volume2D & volume , const Granu :: Callback & callback ) {

	AxisAlignedBoundingBox aabb = volume.getBoundingBox () ;
	if ( aabb.isEmpty () ) return ;
	const int min_x = aabb.getMinX () ;
	const int max_x = aabb.getMaxX () ;
	const int min_y = aabb.getMinY () ;
	const int max_y = aabb.getMaxY () ;

	Matrix < int > lower_height ( min_x , max_x , min_y , max_y ) ;
	for ( int x = max_x ; x >= min_x ; -- x ) {
		int y = min_y ;
		lower_height ( x , y ) = volume.isPointInside ( x , y ) ? 0 : -1 ;
		for ( y = min_y + 1 ; y <= max_y ; ++ y ) {
			if ( ! volume.isPointInside ( x , y ) ) lower_height ( x , y ) = -1 ;
			else lower_height ( x , y ) = lower_height ( x , y - 1 ) + 1 ;
		}
	}

	Matrix < int > upper_height ( min_x , max_x , min_y , max_y ) ;
	for ( int x = max_x ; x >= min_x ; -- x ) {
		int y = max_y ;
		upper_height ( x , y ) = volume.isPointInside ( x , y ) ? 0 : -1 ;
		for ( y = max_y - 1 ; y >= min_y ; -- y ) {
			if ( ! volume.isPointInside ( x , y ) ) upper_height ( x , y ) = -1 ;
			else upper_height ( x , y ) = upper_height ( x , y + 1 ) + 1 ;
		}
	}

	for ( int x = min_x ; x <= max_x ; ++ x ) {
		int max_lower_height = max_y - min_y + 1 ;
		for ( int y = min_y ; y <= max_y ; ++ y ) {
			++ max_lower_height ;
			if ( volume.isPointOnBoundaryInfX ( x , y ) ) {
				int throttled_upper_height = upper_height ( x , y ) ;
				int throttled_lower_height = lower_height ( x , y ) ;
				int width = 1 ;
				while ( x + width <= max_x && volume.isPointInside ( x + width , y ) ) {
					int next_upper = upper_height ( x + width , y ) ;
					int next_lower = lower_height ( x + width , y ) ;
					if ( ( next_upper < throttled_upper_height ) | ( next_lower < throttled_lower_height ) ) {
						if ( throttled_lower_height < max_lower_height ) {
							Rectangle r ( x , x + width - 1 , y - throttled_lower_height , y + throttled_upper_height ) ;
							callback.largestIncludedRectangleFound ( r ) ;
						}
						throttled_upper_height = min ( throttled_upper_height , next_upper ) ;
						throttled_lower_height = min ( throttled_lower_height , next_lower ) ;
					}
					++ width ;
				}
				if ( throttled_lower_height < max_lower_height ) {
					Rectangle r ( x , x + width - 1 , y - throttled_lower_height , y + throttled_upper_height ) ;
					callback.largestIncludedRectangleFound ( r ) ;
				}
				max_lower_height = 0 ;
			}
		}
	}

}

class PushBackRectanglesInVector : public Granu :: Callback {

private :

	vector < Rectangle > & listing ;

public :

	PushBackRectanglesInVector ( vector < Rectangle > & listing ) : listing ( listing ) {}

	~ PushBackRectanglesInVector () {}

	void largestIncludedRectangleFound ( const Rectangle & rectangle ) const {
		listing.push_back ( rectangle ) ;
	}

} ;

void Granu :: addLargestIncludedRectangles ( const Volume2D & volume , vector < Rectangle > & listing ) {
	PushBackRectanglesInVector callback ( listing ) ;
	findLargestIncludedRectangles ( volume , callback ) ;
}

bool Granu :: checkLargestIncludedRectangles ( const Volume2D & volume , const vector < Rectangle > & listing ) {

	cout << "Checking " << listing.size () << " rectangles ..." << endl ;

	for ( unsigned int i = 0 ; i < listing.size () ; ++ i ) {
		for ( unsigned int j = i + 1 ; j < listing.size () ; ++ j ) {
			if ( Rectangle :: equals ( listing [ i ] , listing [ j ] ) ) {
				cerr << "  [ PROBLEM DETECTED ] Rectangles # " << i << " == Rectangle # " << j << endl ;
				return false ;
			}
		}
	}
	cout << "  [ OK ] All rectangles are unique" << endl ;

	vector < const Volume2D * > pointers ;
	for ( unsigned int i = 0 ; i < listing.size () ; ++ i ) {
		pointers.push_back ( & listing [ i ] ) ;
	}
	Volume2D * reconstructed = Volume2D :: createUnion ( pointers ) ;
	if ( ! Volume2D :: equals ( volume , * reconstructed ) ) {
		cerr << "  [ PROBLEM DETECTED ] Some rectangles are missing or too big" << endl ;
		delete reconstructed ;
		return false ;
	}
	delete reconstructed ;
	cout << "  [ OK ] All rectangles are in the silhouette" << endl ;
	cout << "  [ OK ] Each pixel of the volume is covered by at least one rectangle" << endl ;

	for ( unsigned int i = 0 ; i < listing.size () ; ++ i ) {
		Rectangle r = listing [ i ] ;
		Rectangle xinf ( r.getMinX () - 1 , r.getMinX () - 1 , r.getMinY () , r.getMaxY () ) ;
		Rectangle xsup ( r.getMaxX () + 1 , r.getMaxX () + 1 , r.getMinY () , r.getMaxY () ) ;
		Rectangle yinf ( r.getMinX () , r.getMaxX () , r.getMinY () - 1 , r.getMinY () - 1 ) ;
		Rectangle ysup ( r.getMinX () , r.getMaxX () , r.getMaxY () + 1 , r.getMaxY () + 1 ) ;
		if (
				xinf.isIncludedIn ( volume ) ||
				xsup.isIncludedIn ( volume ) ||
				yinf.isIncludedIn ( volume ) ||
				ysup.isIncludedIn ( volume ) ) {
			cerr << "  [ PROBLEM DETECTED ] Rectangle # " << i << " is not maximal" << endl ;
			return false ;
		}
	}
	cout << "  [ OK ] All rectangles are maximum" << endl ;

	cout << "Check is finished." << endl ;
	cout << "N.B. : it is impossible to check if ALL maximum included rectangles are listed" << endl ;
	return true ;

}
