/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009, 2010, 2011 Sébastien PIERARD
 */

#include "Volume2D.h"
#include "Matrix.h"

using namespace std ;

bool Volume2D :: isPointOnBoundaryInfX ( int x , int y ) const {
	return isPointInside ( x , y ) && ! isPointInside ( x - 1 , y ) ;
}

bool Volume2D :: isPointOnBoundarySupX ( int x , int y ) const {
	return isPointInside ( x , y ) && ! isPointInside ( x + 1 , y ) ;
}

bool Volume2D :: isPointOnBoundaryInfY ( int x , int y ) const {
	return isPointInside ( x , y ) && ! isPointInside ( x , y - 1 ) ;
}

bool Volume2D :: isPointOnBoundarySupY ( int x , int y ) const {
	return isPointInside ( x , y ) && ! isPointInside ( x , y + 1 ) ;
}

bool Volume2D :: equals ( const Volume2D & v1 , const Volume2D & v2 ) {

	AxisAlignedBoundingBox aabb1 = v1.getBoundingBox () ;
	AxisAlignedBoundingBox aabb2 = v2.getBoundingBox () ;
	if ( ! AxisAlignedBoundingBox :: equals ( aabb1 , aabb2 ) ) return false ;

	const int min_x = aabb1.getMinX () ;
	const int max_x = aabb1.getMaxX () ;
	const int min_y = aabb1.getMinY () ;
	const int max_y = aabb1.getMaxY () ;

	for ( int x = min_x ; x <= max_x ; ++ x ) {
		for ( int y = min_y ; y <= max_y ; ++ y ) {
			if ( v1.isPointInside ( x , y ) != v2.isPointInside ( x , y ) ) return false ;
		}
	}

	return true ;

}

//----------------------------------------------------------------------//

class VolumeFromImage2D : public Volume2D {

private :

	const GrayscaleImage & img ;
	AxisAlignedBoundingBox * aabb ;

public :

	VolumeFromImage2D ( const GrayscaleImage & img ) : img ( img ) {
		const int w = ( int ) img.getWidth () ;
		const int h = ( int ) img.getHeight () ;
		aabb = new AxisAlignedBoundingBox () ;
		for ( int row = h - 1 ; row >= 0 ; -- row ) {
			for ( int col = w - 1 ; col >= 0 ; -- col ) {
				if ( isPointInside ( col , h - 1 - row ) ) {
					aabb -> extendToIncludePoint ( col , h - 1 - row ) ;
				}
			}
		}
	}

	bool isPointInside ( int x , int y ) const {
		const int w = ( int ) img.getWidth () ;
		const int h = ( int ) img.getHeight () ;
		if ( ( x < 0 ) | ( x >= w ) | ( y < 0 ) | ( y >= h ) ) return false ;
		return img.getPixel ( h - 1 - y , x ) > 127 ;
	}

	AxisAlignedBoundingBox getBoundingBox () const {
		return * aabb ;
	}

	~ VolumeFromImage2D () {
		delete aabb ;
	}

} ;

Volume2D * Volume2D :: createVolumeFromImage ( const GrayscaleImage & img ) {
	return new VolumeFromImage2D ( img ) ;
}

//----------------------------------------------------------------------//

class HollowSphere : public Volume2D {

private :

	int externalRadius ;
	int internalRadiusSq ;
	int externalRadiusSq ;

public :

	HollowSphere ( int internalRadius , int externalRadius ) {
		this -> externalRadius = externalRadius ;
		this -> internalRadiusSq = internalRadius * internalRadius ;
		this -> externalRadiusSq = externalRadius * externalRadius ;
	}

	bool isPointInside ( int x , int y ) const {
		int rSq = x * x + y * y ;
		if ( rSq > externalRadiusSq ) return false ;
		if ( rSq < internalRadiusSq ) return false ;
		return true ;
	}

	AxisAlignedBoundingBox getBoundingBox () const {
		if ( externalRadiusSq >= internalRadiusSq ) {
			return AxisAlignedBoundingBox ( - externalRadius , externalRadius , - externalRadius , externalRadius ) ;
		}
		else {
			return AxisAlignedBoundingBox () ;
		}
	}

	~ HollowSphere () {}

} ;

Volume2D * Volume2D :: createHollowSphere ( int internalRadius , int externalRadius ) {
	return new HollowSphere ( internalRadius , externalRadius ) ;
}

//----------------------------------------------------------------------//

class Union : public Volume2D {

private :

	AxisAlignedBoundingBox * aabb ;
	Matrix < bool > * mat ;

public :

	Union ( const std::vector < const Volume2D * > & listing ) {
		aabb = new AxisAlignedBoundingBox () ;
		for ( unsigned int i = 0 ; i < listing.size () ; ++ i ) {
			aabb -> extendToIncludeAABB ( listing [ i ] -> getBoundingBox () ) ;
		}
		if ( aabb -> isEmpty () ) {
			mat = NULL ;
		}
		else {
			const int min_x = aabb -> getMinX () ;
			const int max_x = aabb -> getMaxX () ;
			const int min_y = aabb -> getMinY () ;
			const int max_y = aabb -> getMaxY () ;
			mat = new Matrix < bool > ( min_x , max_x , min_y , max_y ) ;
			for ( int x = min_x ; x <= max_x ; ++ x ) {
				for ( int y = min_y ; y <= max_y ; ++ y ) {
					( * mat ) ( x , y ) = false ;
				}
			}
			for ( unsigned int i = 0 ; i < listing.size () ; ++ i ) {
				AxisAlignedBoundingBox aabb2 = listing [ i ] -> getBoundingBox () ;
				const int min_x = aabb2.getMinX () ;
				const int max_x = aabb2.getMaxX () ;
				const int min_y = aabb2.getMinY () ;
				const int max_y = aabb2.getMaxY () ;
				for ( int x = min_x ; x <= max_x ; ++ x ) {
					for ( int y = min_y ; y <= max_y ; ++ y ) {
						if ( listing [ i ] -> isPointInside ( x , y ) ) {
							( * mat ) ( x , y ) = true ;
						}
					}
				}
			}
		}
	}

	bool isPointInside ( int x , int y ) const {
		const int min_x = aabb -> getMinX () ;
		const int max_x = aabb -> getMaxX () ;
		const int min_y = aabb -> getMinY () ;
		const int max_y = aabb -> getMaxY () ;
		if ( x < min_x || x > max_x ) return false ;
		if ( y < min_y || y > max_y ) return false ;
		return ( * mat ) ( x , y ) ;
	}

	AxisAlignedBoundingBox getBoundingBox () const {
		return * aabb ;
	}

	~ Union () {
		delete aabb ;
		if ( mat != NULL ) delete mat ;
	}

} ;

Volume2D * Volume2D :: createUnion ( const std::vector < const Volume2D * > & listing ) {
	return new Union ( listing ) ;
}

bool Volume2D :: contains ( const Volume2D & other ) const {
	AxisAlignedBoundingBox this_aabb = this -> getBoundingBox () ;
	AxisAlignedBoundingBox other_aabb = other.getBoundingBox () ;
	if ( this_aabb.getMinX () > other_aabb.getMinX () ) return false ;
	if ( this_aabb.getMaxX () < other_aabb.getMaxX () ) return false ;
	if ( this_aabb.getMinY () > other_aabb.getMinY () ) return false ;
	if ( this_aabb.getMaxY () < other_aabb.getMaxY () ) return false ;
	const int min_x = other_aabb.getMinX () ;
	const int max_x = other_aabb.getMaxX () ;
	const int min_y = other_aabb.getMinY () ;
	const int max_y = other_aabb.getMaxY () ;
	for ( int x = min_x ; x <= max_x ; ++ x ) {
		for ( int y = min_y ; y <= max_y ; ++ y ) {
			if ( other.isPointInside ( x , y ) && ! this -> isPointInside ( x , y ) ) return false ;
		}
	}
	return true ;
}

int Volume2D :: getArea () const {
	AxisAlignedBoundingBox aabb = this -> getBoundingBox () ;
	const int min_x = aabb.getMinX () ;
	const int max_x = aabb.getMaxX () ;
	const int min_y = aabb.getMinY () ;
	const int max_y = aabb.getMaxY () ;
	int area = 0 ;
	for ( int x = min_x ; x <= max_x ; ++ x ) {
		for ( int y = min_y ; y <= max_y ; ++ y ) {
			if ( isPointInside ( x , y ) ) ++ area ;
		}
	}
	return area ;
}
