#include "sema.h"

semStruc semBuf[MAX_SEM];

/*----------------------------------------------------------------
 * getSemValue retourne la valeur d'un smaphore.
 *
 * argument : sem est le numro identifiant le smaphore.
 *
 * valeur de retour : la valeur du smaphore.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
unsigned char getSemValue(unsigned char sem) {
  return semBuf[sem].value;
}


/*----------------------------------------------------------------
 * setSemValue modifie la valeur d'un smaphore.
 *
 * arguments : sem est le numro du smaphore
 *             val est la nouvelle valeur  attribuer au smaphore
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void setSemValue(unsigned char sem, unsigned char val) {
  semBuf[sem].value = val;
}


/*----------------------------------------------------------------
 * higherWaitingTask donne le numro de la tche la plus
 * prioritaire qui est en attente sur un smaphore.
 *
 * argument : sem est le numro du smaphore.
 *
 * valeur de retour : le numro de la tche de priorit la plus
 *                    leve qui attend sur le smaphore.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN AUCUN CAS appeler directement 
 * cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
unsigned char higherWaitingTask(unsigned char sem) {
  return higherTask(semBuf[sem].waitingTasks);
}


/*----------------------------------------------------------------
 * addWaitingTask ajoute une tche  l'ensemble de celles qui
 * attendent sur un smaphore.
 *
 * arguments : sem est le numro du smaphore.
 *             task est le numro de la tche  ajouter.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN AUCUN CAS appeler directement 
 * cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void addWaitingTask(unsigned char sem, unsigned char task) {
  setTaskBit(&(semBuf[sem].waitingTasks), task);
}

/*----------------------------------------------------------------
 * remWaitingTask enlve une tche de l'ensemble de celles qui
 * attendent sur un smaphore.
 *
 * arguments : sem est le numro du smaphore.
 *             task est le numro de la tche  retirer.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN AUCUN CAS appeler directement 
 * cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void remWaitingTask(unsigned char sem, unsigned char task) {
  clearTaskBit(&(semBuf[sem].waitingTasks), task);
}


/*----------------------------------------------------------------
 * waitSem ralise l'opration classique d'attente d'une tche sur
 * un smaphore. Si la valeur du smaphore est nulle, la tche
 * appelante est mise en attente sur ce smaphore. Elle est 
 * ajoute  l'ensemble des tches qui sont dj en attente sur ce
 * smaphore et stoppe. Si la valeur du smaphore n'est pas 
 * nulle, celle-ci est dcrmente et la tche peut poursuivre 
 * son excution.
 *
 * arguments : task est le numro de la tche.
 *             sem est le numro du smaphore.
 * 
 * valeur de retour : aucune.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN AUCUN CAS appeler directement 
 * cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void waitSem(unsigned char task, unsigned char sem) {
  unsigned char semVal;

  if ((semVal = getSemValue(sem)))
    setSemValue(sem, semVal - 1);

  else {
    addWaitingTask(sem, task);
    stopTask(task);
#ifdef WSDEBUG
    printf("Task %d is waiting on semaphore %d\n", task, sem);
#endif
  }
}


/*----------------------------------------------------------------
 * signalSem ralise l'opration classique signal. Si la valeur
 * du smaphore est non nulle, ce qui veut dire qu'aucune tche
 * n'est en attente sur un smaphore, la valeur decelui-ci est
 * simplement incrmente. Si sa valeur est nulle (ce qui veut
 * dire qu'il y a peut-tre une tche en attente sur ce 
 * smaphore), la tche la plus prioritaire en attente sur ce 
 * smaphore, si elle existe, est  nouveau rendue excutable.
 * Si aucune tche n'tait en attente, la tche ayant excut 
 * l'appel poursuit son excution.
 *
 * arguments : sem est le numro du smaphore.
 * 
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void signalSem(unsigned char sem) {
  unsigned char semVal, task;

  task = higherWaitingTask(sem);

  /* si task == MAX_TASK, cela veut dire qu'aucune tche
     n'attend sur le smaphore, donc on incrmente simplement
     sa valeur 
  */

  if ((semVal = getSemValue(sem)) || task == MAX_TASK)
    setSemValue(sem, semVal + 1);

  else {
    restartTask(task);
    remWaitingTask(sem, task);
#ifdef WSDEBUG
    printf("Task %d, previously waiting on semaphore %d, " \
	   "is restarted\n", task, sem);
#endif
  }
}

