#include "kernel.h"
#include <stdlib.h> /* exit() */

const unsigned char bitPower[] = {1, 2, 4, 8, 16, 32, 64, 128};

#if MAX_TASK == 64
prtyStruc runningTasks = {0, {0, 0, 0, 0, 0, 0, 0, 0}};
#else
prtyStruc runningTasks = {0};
#endif

taskStruc taskBuf[MAX_TASK];

#ifdef HI_TECH_C
#pragma interrupt_level 0
#endif
/*----------------------------------------------------------------
 * setTaskBit positionne le bit correspondant  une tche dans une
 * structure de stockage de priorits de tches.
 *
 * arguments : prtyTblPtr est un pointeur vers la structure de
 *             priorits  modifier.
 *             task est le numro de la tche dont on doit 
 *             positionner le bit correspondant.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN PRINCIPE PAS appeler directement 
 * cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
void setTaskBit(prtyStruc *prtyTblPtr, unsigned char task) {
#if MAX_TASK == 64
  unsigned char posY;

  posY = task >> 3;
  prtyTblPtr->x[posY] |= bitPower[task & 7];
  prtyTblPtr->y |= bitPower[posY];
#else
  prtyTblPtr->x |= bitPower[task & 7];
#endif
}


/*----------------------------------------------------------------
 * clearTaskBit efface le bit correspondant  une tche dans une
 * structure de stockage de priorits de tches.
 *
 * arguments : prtyTblPtr est un pointeur vers la structure de
 *             priorits  modifier.
 *             task est le numro de la tche dont on doit 
 *             effacer le bit correspondant.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN PRINCIPE PAS appeler directement 
 * cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
void clearTaskBit(prtyStruc *prtyTblPtr, unsigned char task) {
#if MAX_TASK == 64
  unsigned char posY;

  posY = task >> 3;

  if (!(prtyTblPtr->x[posY] &= ~bitPower[task & 7]))
    prtyTblPtr->y &= ~bitPower[posY];
#else
  prtyTblPtr->x &= ~bitPower[task & 7];
#endif
}

/*----------------------------------------------------------------
 * getTaskBit renvoie 0 si le bit correspondant  une tche
 * est positionn dans une structure de priotit de tche.
 *
 * arguments : prtyTblPtr est un pointeur vers la structure de
 *             priorits.
 *             task est le numro de la tche. 
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN PRINCIPE PAS appeler directement 
 * cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
unsigned char getTaskBit(prtyStruc *prtyTblPtr, unsigned char task) {
#if MAX_TASK == 64
  unsigned char posY;

  posY = task >> 3;

  return (prtyTblPtr->x[posY] & bitPower[task & 7]);
#else
  return (prtyTblPtr->x & bitPower[task & 7]);
#endif
}


/*----------------------------------------------------------------
 * higherTask donne le numro de la tche la plus prioritaire dont
 * le bit est positionn dans la structure de stockage de
 * priorits de tches donne.
 *
 * argument : taskPrty est la structure de priorits  consulter.
 *
 * valeur de retour : le numro de la tche de priorit la plus
 *                    leve dont le bit est positionn dans la
 *                    table.
 *
 * note : cette fonction est utilise de manire interne par le 
 * kernel. Les tches et les fonctions de traitement
 * d'interruption NE PEUVENT EN PRINCIPE PAS appeler directement 
 * cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
unsigned char higherTask(prtyStruc tasksPrty) {
#if MAX_TASK == 64
  char posY, posX;

  for(posY = 7; posY >= 0; posY--)
    if (tasksPrty.y & bitPower[(unsigned char) posY]) break;

  if (posY < 0) return MAX_TASK;   /* aucune tche n'est active? */

  for(posX = 7;; posX--)           /* le gardien est inutile ici
				      car il doit y avoir une
				      tche active si l'excution
				      se poursuit jusqu'ici */
    if (tasksPrty.x[(unsigned char) posY] & 
	bitPower[(unsigned char) posX]) break;

  return (posX + (posY << 3));
#else
  char posX;

  for(posX = 7; posX >= 0; posX--)    
    if (tasksPrty.x & bitPower[(unsigned char) posX]) break;

  if (posX < 0) return MAX_TASK;   /* aucune tche n'est active? */
  return posX;
#endif
}


#ifdef HI_TECH_C
#pragma interrupt_level 0
#endif
/*----------------------------------------------------------------
 * startTask indique au kernel qu'une tche devient excutable.
 * Cette tche verra ses instructions excutes par le processeur
 * quand elle sera la tche excutable la plus prioritaire 
 * instancie sur le systme. FIX_ME completer
 *
 * argument : task est le numro de la tche.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
void startTask(unsigned char task) {

 if (!getTaskBit(&runningTasks, task)) {
   taskBuf[task].pos = 0;
   setTaskBit(&runningTasks, task);
 }
}


/*----------------------------------------------------------------
 * killTask indique au kernel qu'une tche n'est plus excutable.
 * Cette fonction retire aussi la tche de tous les ensembles de
 * tches en attente sur les smaphores et rinitialise sa 
 * position courante d'excution au dbut de son code.
 *
 * argument : task est le numro de la tche.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
void killTask(unsigned char task) {

#ifdef USE_SEMAPHORE
  unsigned char i;
#endif

  clearTaskBit(&runningTasks, task);
  taskBuf[task].pos = 0;

#ifdef USE_SEMAPHORE
  /* code de suppression de la tche dans les ensembles de 
     tches en attente sur les smaphores */
  for(i = 0; i < MAX_SEM; i++) remWaitingTask(i, task);
#endif
}


/*----------------------------------------------------------------
 * stopTask indique au kernel qu'une tche n'est plus excutable.
 * La position courante d'excution de la tche ainsi que les
 * informations concernant l'ventuelle attente de cette tche
 * sur des smaphores ne sont pas modifies par cette fonction.
 *
 * argument : task est le numro de la tche.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void stopTask(unsigned char task) {
  clearTaskBit(&runningTasks, task);
}


/*----------------------------------------------------------------
 * restartTask rend  nouveau excutable une fonction qui a
 * prcdemment t stoppe par la fonction stopTask.
 *
 * argument : task est le numro de la tche.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction peut tre appele dans une fonction de 
 * traitement d'interruption. Elle est principalement utilise de
 * manire interne par le kernel. Les tches NE devraient PAS
 * appeler directement cette fonction.
 * type : facultative.
 *----------------------------------------------------------------
 */
void restartTask(unsigned char task) {
  setTaskBit(&runningTasks, task);
}


/*----------------------------------------------------------------
 * schedule est la fonction la plus importante du kernel.
 * Lorsqu'elle reprend le contrle du systme, elle slectionne
 * la tche excutable la plus prioritaire et relance son 
 * excution. De plus, elle ralise effectivement l'appel systme
 * demand par une tche.
 *
 * argument : aucun.
 *
 * valeur de retour : aucune.
 *
 * note : cette fonction NE DOIT ETRE APPELEE QU'UNE SEULE FOIS,
 * dans la fonction main de l'application. Le mcanisme qui permet
 * de lui rendre le contrle pour lui faire excuter des appels
 * systmes est expliqu en dtail dans le rapport.
 * type : ncessaire.
 *----------------------------------------------------------------
 */
void schedule() {
  unsigned char task, sysCallCode;
  taskParmStruc *taskParmPtr;

  for(;;) {

    /* dans une vraie application, on boucle indfiniment si il n'y
       a pas de tache active mais pour les tests sur une station de 
       travail, on s'arrete...
    */

#ifdef WSDEBUG
    if ((task = higherTask(runningTasks)) == MAX_TASK) break;
#else
    if ((task = higherTask(runningTasks)) == MAX_TASK) continue;
#endif

    /* en principe, on ne peut pas interdire les interruptions ici
       mais si les tches utilisent les huit niveaux de la pile,
       plus aucune interruption ne pourra se dclencher sans mener
        un crash du systme... (cas du PIC)
    */

    di(); /*  enlever */

    taskParmPtr = dispatch(task, taskBuf[task].pos);

    ei(); /*  enlever */


    taskBuf[task].pos = taskParmPtr->pos;
    sysCallCode = taskParmPtr->sysCallCode;

    switch(sysCallCode) {
    case RESCHEDULE:
      break;

    case TASK_START:
      di();
      startTask(taskParmPtr->task);
      ei();
      break;

    case TASK_EXIT:
      di();
      killTask(taskParmPtr->task);
      ei();
      break;

#ifdef USE_SEMAPHORE
    case SIGNAL_SEM:
      di();
      signalSem(taskParmPtr->sem);
      ei();
      break;
	      
    case WAIT_SEM:
      waitSem(task, taskParmPtr->sem);
      break;
	      
    case SET_SEM_VALUE:
      di();
      setSemValue(taskParmPtr->sem, taskParmPtr->val);
      ei();
      break;
	      
    case GET_SEM_VALUE:
      di();
      taskParmPtr->val = getSemValue(taskParmPtr->sem);
      ei();
      break;
#endif                         /* #ifdef USE_SEMAPHORE */

#ifdef WSDEBUG
    default:
      printf("unknown syscall code %d\n", sysCallCode);
      exit(-1);
#endif
    }
  }
}
