/****************************************************************************/
/**                                                                        **/
/**  ev.c  : Evaluateur simple.                                            **/
/**  ----                                                                  **/
/**                                                                        **/
/**  B. Boigelot, 10/94.                                                    **/
/**                                                                        **/
/****************************************************************************/

/****************************************************************************

  Mode d'emploi
  =============

  a) #include "ev.h"          pour definir les symboles, les codes de retour,
                              et le type lex_unit.
  
  b) ev_init();               pour initialiser l'evaluateur (c'est-a-dire le
                              preparer a evaluer une nouvelle expression).

  c) lex_unit      token;
     unsigned long value;
     int           status;

     token.symbol = ... ;
     token.value  = ... ;

     status = ev_token(&token, &value);

                              pour passer un nouvel element lexical 'token'
                              a l'evaluateur. Le champ 'token.symbol' doit
                              contenir le code de l'element lexical concerne
                              (CONST, LPAR, ...) ou END pour signifier la fin
                              de la sequence d'entree. Si l'element lexical
                              est une constante (CONST), le champ
                              'token.value' doit contenir la valeur de cette
                              constante. 'ev_token' renvoie ERR en cas
                              d'erreur de syntaxe ou de depassement, EXC en
                              de division par zero, VAL si le resultat de
                              l'evaluation est disponible dans 'value', et
                              OK dans toutes les autres situations.

  d) cc -o mon-programme mon-programme.c ev.o

                              pour lier au programme le code objet de
                              l'evaluateur.

*****************************************************************************/

#include "ev.h"

/*** Definitions ***/

#define ERROR          0
#define SHIFT          1
#define REDUCE         2

#define NB_SYMB       20
#define NB_TERM       12
#define NB_PROD       18

#define MAX_STACK   1024

/*** Tables internes derivees de la grammaire ***/

static char l_table[NB_SYMB] = 
 { 0, 1, 0, 1, 0, 0, 2, 0, 0, 0, 0, 0, 3, 4, 5, 6, 0, 1, 0, 7 };

static unsigned char sr_table[][(NB_TERM + 3) / 4] =
 { {  20, 16,  0 }, { 130, 170, 170 }, {  40,  32,   0 }, { 130, 1, 0 },
   { 130,  2, 80 }, { 130,  22, 160 }, { 130, 106, 165 }, {  64, 0, 0 } };

static unsigned char start_rule[NB_SYMB] =
 { 0, 0, 0, 1, 0, 0, 2, 0, 0, 0, 0, 0, 4, 5, 7, 10, 0, 13, 0, 0 };

static unsigned char end_rule[NB_SYMB] =
 { 0, 0, 0, 1, 0, 0, 3, 0, 0, 0, 0, 0, 4, 6, 9, 12, 0, 17, 0, 0 };

static unsigned char derivation[NB_PROD][2] =
 { { 0,  0 }, { 0, 19 }, {  0,  0 }, {  0, 14 }, {  0, 2 }, {  0,  0 },
   { 12, 4 }, { 0,  0 }, { 13, 10 }, { 13, 11 }, {  0, 0 }, { 14,  5 },
   { 0, 16 }, { 0,  0 }, { 15,  7 }, { 15,  8 }, { 15, 9 }, {  0, 18 } };

static unsigned char first_symbol[NB_PROD] =
 { 17, 17, 18, 16, 19, 12, 12, 13, 13, 13, 14, 14, 14, 15, 15, 15, 15, 17 };

static unsigned char rule_len[NB_PROD] =
 { 1, 2, 1, 2, 2, 1, 3, 1, 3, 3, 1, 3, 2, 1, 3, 3, 3, 2 };

/*** Regles de production (symbole distingue : <expr>) ***

0)  <expr4>          ::= CONST
1)  <expr4>          ::= <par> ')'
2)  <minus>          ::= '-'
3)  <expr2b>         ::= <expr2> '-'
4)  <par>            ::= '(' <expr>
5)  <expr>           ::= <expr1>
6)  <expr>           ::= <expr> '&' <expr1>
7)  <expr1>          ::= <expr2>
8)  <expr1>          ::= <expr1> LSHIFT <expr2>
9)  <expr1>          ::= <expr1> RSHIFT <expr2>
10)  <expr2>         ::= <expr3>
11)  <expr2>         ::= <expr2> '+' <expr3>
12)  <expr2>         ::= <expr2b> <expr3>
13)  <expr3>         ::= <expr4>
14)  <expr3>         ::= <expr3> '*' <expr4>
15)  <expr3>         ::= <expr3> '/' <expr4>
16)  <expr3>         ::= <expr3> '%' <expr4>
17)  <expr4>         ::= <minus> <expr4>

 *** Macros ***/

#define get_entry(s1, s2)   ((sr_table[l_table[s1]][(s2)>>2]>>(2*((s2)%4)))%4)

#define stack_pick(depth)    stack[stack_pointer - (depth)]
#define stack_top()         (stack_pick(1))
#define stack_remove(n)     (stack_pointer -= (n))
#define stack_push(x)       (stack[stack_pointer++] = *(x))

/*** Variables globales ***/

static unsigned stack_pointer;
static lex_unit stack[MAX_STACK];

/*** Fonctions ***/

/******
  Application de la fonction semantique associee a une regle syntaxique.
******/

unsigned long semantics(rule, ok)
  unsigned int rule, *ok;
{
  *ok = 1;
  switch(rule) {
    case  0 :
    case  2 :
    case  4 :
    case  5 :
    case  7 :
    case 10 :
    case 13 : return stack_top().value;
    case  6 : return stack_pick(3).value & stack_top().value;
    case  8 : return stack_pick(3).value << stack_top().value;
    case  9 : return stack_pick(3).value >> stack_top().value;
    case 11 : return stack_pick(3).value + stack_top().value;
    case 12 : return stack_pick(2).value - stack_top().value;
    case 14 : return stack_pick(3).value * stack_top().value;
    case 15 : return (*ok = !!stack_top().value) ?
                  stack_pick(3).value / stack_top().value : 0L;
    case 16 : return stack_pick(3).value % stack_top().value;
    case 17 : return (unsigned long) (-stack_top().value);
    case  1 : 
    case  3 : return stack_pick(2).value;
    }
}

/******
  Recherche efficace de la regle de production impliquee dans une operation
  de reduction.
******/

unsigned int search_rule() {

  unsigned char s0, stack_symbol1, stack_symbol2, test_symbol;
  unsigned int first_pointer, start_pointer, end_pointer, test_pointer;

  s0            = stack_top().symbol;
  first_pointer = start_rule[s0];
  if (!(stack_symbol1 = stack_pick(2).symbol))
    return first_pointer;
  stack_symbol2 = stack_pick(3).symbol;
  start_pointer = first_pointer;
  end_pointer   = end_rule[s0];
  do {
    test_pointer = (start_pointer + end_pointer + 1) / 2;
    switch(rule_len[test_pointer]) {
      case 1 : return first_pointer;
      case 2 : if ((test_symbol = derivation[test_pointer][1]) ==
                   stack_symbol1)
                 return test_pointer;
               if (test_symbol > stack_symbol1)
                 end_pointer = test_pointer - 1;
               else
                 start_pointer = test_pointer + 1;
               break;
      case 3 : if ((test_symbol = derivation[test_pointer][1]) ==
                   stack_symbol1) {
                 if (!stack_symbol2) 
                   return first_pointer;
                 if ((test_symbol = derivation[test_pointer][0]) ==
                     stack_symbol2)
                   return test_pointer;
                 if (test_symbol > stack_symbol2)
                   end_pointer = test_pointer - 1;
                 else
                   start_pointer = test_pointer + 1;
                 } 
              else 
                if (test_symbol > stack_symbol1)
                  end_pointer = test_pointer - 1;
                else
                  start_pointer = test_pointer + 1;
      }
    } while (start_pointer <= end_pointer);
  return first_pointer; 
}

/******
  Initialisation de l'evaluateur.
 ******/

void ev_init() {
  stack_pointer   = 1;
  stack[0].symbol = 0;
  stack[0].value  = 0L; 
}

/*****
  Analyse d'un element lexical.
 *****/

int ev_token(token, value)
  lex_unit      *token;
  unsigned long *value;
{
  unsigned char stack_symbol, strip_symbol;
  unsigned int  rule, ok;
  lex_unit      new;

  for (ok = 1, strip_symbol = token -> symbol ; ok;) {
    stack_symbol = stack_top().symbol;  
    if (stack_pointer == 2 && stack_symbol == NB_TERM) {
     *value = stack_top().value;    
      return VAL;
      }
    switch(get_entry(stack_symbol, strip_symbol)) {  
      case ERROR  : return ERR;
      case SHIFT  : ok   = 0;
                    new  = *token;
                    break;
      case REDUCE : rule = search_rule();
                    new.symbol = first_symbol[rule];
                    new.value  = semantics(rule, &ok);
                    if (!ok)
                      return EXC;
                    stack_remove(rule_len[rule]);
      }
    if (stack_pointer >= MAX_STACK)
      return ERR;
    stack_push(&new);  
    }
  return OK;
}
