/* 
 * Fichier:hard.c
 * Emulation de MAYBE au niveau lmentaire (nanocode)
 */

/* On ne peut pas utiliser memset sur un tableau de 64K bytes si memset
 * prend un int de 16 bits comme argument, ce qui est le cas avec MSC7.
 * Si la rom atteignait 64k, il faudrait donc transformer la ligne qui suit
 * en commentaire.
 */
#define USEMEMSET

/*
 * Il y a deux mthodes pour simuler le transfert de registres:
 * La premire analyse le mot de slection (par un switch),
 * la seconde utilise une table de fonctions.
 * Selon la machine, l'une ou l'autre est plus rapide.
 * Si USERTLTABLE est dfini, on utilise la table.
 */
#define USERTLTABLE

#include <stdio.h>

#ifdef USEMEMSET
#include <memory.h>
#endif

#include "maysim.h"
#include "machine.h"	/* pour Refresh */
#include "hard.h"

#if TRUE
#include <stdlib.h>
#include <string.h>
#include "display.h"
#endif

/*********************
 * Memoires diverses *
 *********************/

FILE *fd;
char msg[128];

unsigned int CtlRom[CtlRomSize];             /* Roms de controle (2764)      */
byte *CtlRomTop;
byte OpReg,PhaseReg;                         /* Adresses nanocode en deux    */
                                             /* parties, 8+4bits, 74LS377    */
                                             /* et 74LS163                   */

byte MicroRom[MicroRomSize];                 /* Rom de microcode (2764)      */
byte *MicroRomTop;


unsigned int PCReg,PCTrack;                  /* Program counter de microcode */
                                             /* (13 bits utiles)             */

/* Ram statique (registres soft) */
byte SRam[SRamSize];                         /* 6116 utilise partiellement   */
byte MAReg;                                  /* latch octuple 74LS377        */
byte *SRamTop;

/* Ram dynamique */
byte DRam[DRamSize];
byte *DRamTop;
unsigned int DRamAddr;                    /* latch interne RAS/CAS des DRams */
int DRamPhase=-1,DRamCount; /* Information de timing pour simuler le 74LS194 */
/* La generation d'un RAS, d'un CAS ou d'un WR s'effectue au moyen du meme
 * signal de controle LDDRAM et on ne peut les distinguer par simple
 * examen d'un mot de la rom de controle. Il faut tenir compte de l'historique.
 * En effet, un CAS n'est genere que si un RAS l'a ete au cycle precedent,
 * et un WR si un CAS l'a ete au cycle precedent. Il ne suffit pas d'examiner
 * les mots precedents de la rom de contrele car il pourrait y avoir des
 * variantes de comportement selon le bit de condition. Il faut reellement
 * garder la trace de ce qui s'est passe. Une solution simple mais inefficace
 * serait de gerer un compteur que toutes les nano-instructions remettraient a
 * zero sauf celles qui generent un LDDRAM. Une solution plus rapide est
 * de memoriser la phase du dernier acces DRAM a l'interieur d'une micro-
 * instruction, et de l'invalider seulement au changement de micro-instruction.
 * Vu que la phase ne peut que croitre au cours d'une meme micro-instruction,
 * il n'en resulte aucune perte d'information.
 */
 

byte Switches,Buttons;                     /* Interrupteurs DIL et poussoirs */

                               /* Affichage des erreurs etendu en mode batch */
void ALUFunctions();

void RunError(char *errmsg)
{
  if(BatchMode)
    fprintf(stderr,"\nError at PC=%4.4X, OP=%2.2X, Phase=%1X:",
	    PCReg,OpReg,PhaseReg);
  Error(errmsg);
}

/*****************************
 * Entree/sortie (UART 8250) *
 *****************************/
 
/* Registres accessibles par programme */
byte RBR,THR,IER,IIR,LCR,MCR,LSR,MSR,DivLatchLo,DivLatchHi;
/* Etat simul des pins en entre */
byte CTS,DSR,RI,RLSD;
/* Un flag supplmentaire mmorise la transition active sur THRE */
/* Ncessaire car l'effet de bord de IIR en lecture n'affecte pas THRE */
boolean THREFlag;
/* Mmorisation des sorties */
char OutBuf[TW+1];
int OutCol=0;

/* Accesseurs */
/* Mmorisation des sorties */
char *FirstLinePtr, *LastLinePtr=0;
void AddLine(char *line){
#if FALSE
	char *lptr;
	int len;
	len=strlen(line);
	lptr=malloc(sizeof(LastLinePtr)+len+1);
	if( lptr==0 ){
		 Error("Out of memory");
	}else{
		*(char **)LastLinePtr=lptr;
		LastLinePtr=lptr;
		strcpy(lptr+sizeof(LastLinePtr),line);
	}
#endif
}

/* Ecriture dans les registres internes */
void WriteUart(byte RegNum,byte Source){
/* Effets de bord non implments ?? */
	switch(RegNum){
		case 0x00:
			if (LCR&0x80) /* Bit DLAB */
				DivLatchLo=Source;
			else
				LSR&=0xDF; /* Remet THRE  zro */
				THREFlag=FALSE;
				THR=Source;
				OutBuf[OutCol++]=Source;
				if( Source='\n' || OutCol>TW ){
					OutBuf[OutCol]=0;
					AddLine(OutBuf);
					OutCol=0;
				}
			break;
		case 0x01:
			if (LCR&0x80) /* Bit DLAB */
				DivLatchHi=Source;
			else
				IER=Source&0x0F;
			break;
		case 0x02:
			RunError("IIR:cannot write");
			break;
		case 0x03:
			LCR=Source;
			break;
		case 0x04:
			MCR=Source&0x1F;
			break;
		case 0x05:
			/* Seul le bit 0 est accessible en criture */
			LSR=(LSR&0xFE)|(Source&1);
			break;
		case 0x06:
			RunError("MSR:cannot write"); /* ?? tester */
			break;
		case 0x07:
			RunError("Unimplemented UART register 07");
			break;
		default:
			RunError("Internal error - invalid UART register");
	}
}
byte ReadUart(byte RegNum){
/* Effets de bord non implments ?? */
	byte b;
	switch(RegNum){
		case 0x00:
			if (LCR&0x80) /* Bit DLAB */
				return(DivLatchLo);
			else
				LSR&=0xFE; /* Data ready  zro */
				return(RBR);
		case 0x01:
			if (LCR&0x80) /* Bit DLAB */
				return(DivLatchHi);
			else
				return(IER);
		case 0x02:
			THREFlag=FALSE;
			return(IIR);
		case 0x03:
			return(LCR);
		case 0x04:
			return(MCR);
		case 0x05:
			b=LSR;
			LSR&=0xE1; /* Remet  zro les bits d'erreur */
			return(b);
		case 0x06:
			b=MSR;
			MSR&=0xF0; /* Remet  zro les bits 'delta' */
			if( MCR&32 ) /* Mode de test 'LOOP' */
				b=(b&0x0F)
				  |((MCR&1)<<5)|((MCR&2)<<3)|((MCR&0x0C)<<4);
			return(b);
		case 0x07:
			RunError("Unimplemented UART register 07");
			return(0xFF);
		default:
			RunError("Internal error - invalid UART register");
	}
}

boolean IOFlag(){ /* Vrai si interruption demande */
	return( (IER&LSR&1)		/* Donne reue */
		|| THREFlag		/* Donne envoye */
		|| (IER&4 && LSR&30)	/* Erreur de rception */
		|| (IER&8 && MSR&15) );	/* Changement sur le modem */
}

void ResetUart(){
	IER=LCR=MCR=0;
	IIR=1;
	LSR=0x60;
	MSR=(CTS<<4)+(DSR<<5)+(RI<<6)+(RLSD<<7);
	if( LastLinePtr==0) LastLinePtr=(char *)&FirstLinePtr;
}



/* ALUreg correspond au registre de sortie de l'ALU                          */
int AReg,BReg,ALUReg;
/* CCode represente les lignes A..H du 74LS166
 * tandis que CCodeReg represente son etat interne
 * Ils ne coincident qu'apres un 'Load Condition Code'
 */
byte CCode,CCodeReg;
/* Traduction des codes d'operation de l'ALU en notation symbolique */
char *ALUOpName[48]={
/* Fonctions arithmetiques, carry in = 0 */
"A+1","(A|B)+1","(A|~B)+1","0","A+(A&~B)+1","(A|B)+(A&~B)+1","A-B","(A&B)",
"A+(A&B)+1","A+B+1","(A|~B)+(A&B)+1","A&B","A+A+1","(A|B)+A+1","(A|~B)+A+1","A"
,
/* Fonctions arithmetiques, carry in = 1 */
"A","A|B","A|~B","-1","A+(A&~B)","(A|B)+(A&~B)","A-B-1","(A&B)-1",
"A+(A&B)","A+B","(A|~B)+(A&B)","(A&B)-1","A+A","(A|B)+A","(A|~B)+A","A-1",
/* Fonctions logiques */
"~A","~(A|B)","~A&B","0","~(A&B)","~B","A^B","A&~B",
"~A|B","~(A^B)","B","A&B","-1","A|~B","A|B","A"
};

/* Effectue l'operation specifiee par le byte de controle donne */
void UpdateCCode(){/*
  CCode=(IOFlag()<<6)      
             +Buttons                 
	     +((ALUReg<<3)&8)         
	     +(((ALUReg&0xFF)==0xFF)<<2)   
	     +(((ALUReg>>7)&3)^2); 
	     */
}
 
/* Effectue l'operation specifiee par le mot de controle courant */
void UpdateALU(){
	UpdateCCode();
}

/***************************************
 * Fonctions de transfert de registres *
 ***************************************/

/* Tableau des breakpoints par opcode. Si l'element n n'est pas nul,
 * l'execution s'arretera juste avant l'execution de l'instruction
 * micro-code dont le code machine est n.
 */
int BreakOp[256];

static unsigned long RemainingSteps;

/* Nombre de micro-steps apres lequel on change l'affichage en mode GO
 * Si on le modifie,le raccourci pris pour l'affichage doit etre modifie aussi.
 */
#define STILLALIVE	(1000)

#ifdef USERTLTABLE
/* Definition du tableau des fonctions de transfert de registres */

#pragma check_stack( off )
 
static	unsigned int steps=STILLALIVE,tsteps=0,ops5;
static  unsigned int phasereg,CtlWord;

#define SRCERR	RunError("Source not connected");\
		RemainingSteps=1;


/* Famille 0: destination = Opcode */


#define OPDEST	ops5=OpReg<<5;\
		phasereg=0;\
		DRamPhase=-1;\
		PCTrack=PCReg;\
		if(BreakOp[OpReg]) RemainingSteps=1;\
		if(steps--) return;\
		steps=STILLALIVE;tsteps++;\
		if( BatchMode ){\
			putchar('.');\
			return;\
		}\
		ultod((long)tsteps,MsgBuf);\
		strcat(MsgBuf,"000 micro-steps.");\
		PutStatus(MsgBuf);\
		Refresh();

void ls00(){
	OpReg=Switches;
	OPDEST
}
void ls10(){
	OpReg=MicroRom[PCReg&(MicroRomSize-1)];
	OPDEST
}
void ls20(){
	OpReg=ALUReg;
	OPDEST
}
void ls30(){
	SRCERR
	OpReg=0xFF;
	OPDEST
}
void ls40(){
	OpReg=SRam[MAReg];
	OPDEST
}
void ls50(){
	SRCERR
	OpReg=0xFF;
	OPDEST
}
void ls60(){
	OpReg=DRam[DRamAddr];
	OPDEST
}
void ls70(){
	OpReg=ReadUart((byte)(AReg&7));
	OPDEST
}


/* Famille 1: destination = adresse microrom (PC) */

void ls01(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+Switches);
}
void ls11(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+MicroRom[PCReg&(MicroRomSize-1)]);
}
void ls21(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+ALUReg);
}
void ls31(){
	SRCERR
	PCReg=(MicroRomSize-1)&((PCReg<<8)+0xFF);
}
void ls41(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+SRam[MAReg]);
}
void ls51(){
	SRCERR
	PCReg=(MicroRomSize-1)&((PCReg<<8)+0xFF);
}
void ls61(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+DRam[DRamAddr]);
}
void ls71(){
	PCReg=(MicroRomSize-1)&((PCReg<<8)+ReadUart((byte)(AReg&7)));
}

/* Famille 2: destination = registre A */
void ls02(){
	AReg=Switches;
}
void ls12(){
	AReg=MicroRom[PCReg&(MicroRomSize-1)];
}
void ls22(){
	AReg=ALUReg&0xFF;
}
void ls32(){
	SRCERR
	AReg=0xFF;
}
void ls42(){
	AReg=SRam[MAReg];
}
void ls52(){
	SRCERR
	AReg=0xFF;
}
void ls62(){
	AReg=DRam[DRamAddr];
}
void ls72(){
	AReg=ReadUart((byte)(AReg&7));
}

/* Famille 3: destination = registre B */
void ls03(){
	BReg=Switches;
}
void ls13(){
	BReg=MicroRom[PCReg&(MicroRomSize-1)];
}
void ls23(){
	BReg=ALUReg&0xFF;
}
void ls33(){
	SRCERR
	BReg=0xFF;
}
void ls43(){
	BReg=SRam[MAReg];
}
void ls53(){
	SRCERR
	BReg=0xFF;
}
void ls63(){
	BReg=DRam[DRamAddr];
}
void ls73(){
	BReg=ReadUart((byte)(AReg&7));
}

/* Famille 4: destination = registre en ram statique */
void ls04(){
	SRam[MAReg]=Switches;
}
void ls14(){
	SRam[MAReg]=MicroRom[PCReg&(MicroRomSize-1)];
}
void ls24(){
	SRam[MAReg]=ALUReg;
}
void ls34(){
	SRCERR
	SRam[MAReg]=0xFF;
}
void ls44(){
/*	SRam[MAReg]=SRam[MAReg]; Ceci est en fait un NOOP */
}
void ls54(){
	SRCERR
	SRam[MAReg]=0xFF;
}
void ls64(){
	SRam[MAReg]=DRam[DRamAddr];
}
void ls74(){
	SRam[MAReg]=ReadUart((byte)(AReg&7));
}

/* Famille 5: destination = pointeur de ram statique MAReg */
void ls05(){
	MAReg=Switches;
}
void ls15(){
	MAReg=MicroRom[PCReg&(MicroRomSize-1)];
}
void ls25(){
	MAReg=ALUReg;
}
void ls35(){
	SRCERR
	MAReg=0xFF;
}
void ls45(){
	MAReg=SRam[MAReg];
}
void ls55(){
	SRCERR
	MAReg=0xFF;
}
void ls65(){
	MAReg=DRam[DRamAddr];
}
void ls75(){
	MAReg=ReadUart((byte)(AReg&7));
}

/* Famille 6: destination = ram dynamique */
#define	DRAM(source)	\
	if( DRamPhase==phasereg ){ /* Accs conscutifs */\
		DRamPhase++;\
		switch(DRamCount++){\
			case 0: /* Deuxime accs, RAS */\
				DRamAddr+=(source)<<8;\
				break;\
			case 1: /* Troisime accs, WR */\
				DRam[DRamAddr]=(source);\
				break;\
		/* S'il y a plus de trois accs conscutifs,\
		 * rien ne se passe. En effet, le 74LS194\
		 * reste  0000, et ni CAS ni RAS ni WR ne\
		 * prsententent de flanc descendant.\
		 */\
		}\
	}else{ /* Premier accs, CAS */\
		DRamAddr=(source);\
		DRamCount=0;\
		DRamPhase=phasereg+1;\
	}
	
void ls06(){
	DRAM(Switches)
}
void ls16(){
	DRAM(MicroRom[PCReg&(MicroRomSize-1)])
}
void ls26(){
	DRAM(ALUReg)
}
void ls36(){
	SRCERR
	DRAM(0xFF)
}
void ls46(){
	DRAM(SRam[MAReg])
}
void ls56(){
	SRCERR
	DRAM(0xFF)
}
void ls66(){
	DRAM(DRam[DRamAddr])
}
void ls76(){
	DRAM(ReadUart((byte)(AReg&7)))
}

/* Famille 7: destination = UART */
void ls07(){
	WriteUart((byte)(AReg&7),Switches);
}
void ls17(){
	WriteUart((byte)(AReg&7),MicroRom[PCReg&(MicroRomSize-1)]);
}
void ls27(){
	WriteUart((byte)(AReg&7),(byte)ALUReg);
}
void ls37(){
	SRCERR
	WriteUart((byte)(AReg&7),0xFF);
}
void ls47(){
	WriteUart((byte)(AReg&7),SRam[MAReg]);
}
void ls57(){
	SRCERR
	WriteUart((byte)(AReg&7),0xFF);
}
void ls67(){
	WriteUart((byte)(AReg&7),DRam[DRamAddr]);
}
void ls77(){
	RunError("UART cannot be read and written to simultaneously");
	RemainingSteps=1;
/*	fclose(fd); exit(0); */
	/* WriteUart((byte)(AReg&7),ReadUart((byte)(AReg&7))); */

}

typedef void (LoadFun)();
LoadFun *LoadFuns[64]={
	ls00,ls01,ls02,ls03,ls04,ls05,ls06,ls07,
	ls10,ls11,ls12,ls13,ls14,ls15,ls16,ls17,
	ls20,ls21,ls22,ls23,ls24,ls25,ls26,ls27,
	ls30,ls31,ls32,ls33,ls34,ls35,ls36,ls37,
	ls40,ls41,ls42,ls43,ls44,ls45,ls46,ls47,
	ls50,ls51,ls52,ls53,ls54,ls55,ls56,ls57,
	ls60,ls61,ls62,ls63,ls64,ls65,ls66,ls67,
	ls70,ls71,ls72,ls73,ls74,ls75,ls76,ls77
};

#pragma check_stack( on )

#endif




/******************************************************************************
 * Imlementation du diagramme logic du 74181 IC de la fig. 5.9 p.115,         *
 *  computation structure.                                                    *
 *                                                                            *
 * CONVENTION :                                                               *
 * la numerotation des portes est faite de haut en bas et de niveau a niveau  *
 * (du niveau le plus bas au niveau le plus haut) suivant le diagramme logic. *
******************************************************************************/


int
first_level(a, b, S3, S2, S1, S0)
     int a;
     int b;
     int S3;
     int S2;
     int S1;
     int S0;
{
  int and0   = 0;
  int and1   = 0;
  int and2   = 0;
  int and3   = 0;
  int and4   = 0;
  int nor0   = 0;
  int nor1   = 0;
  int result = 0;

  and0 = a & S3 & b;
  and1 = a & S2 & (~b & 1);
  nor0 = ~(and0 | and1) & 1;

  and2 = (~b & 1) & S1;
  and3 = S0 & b;
  and4 = a & 1;
  nor1 = ~(and2 | and3 | and4) & 1;

  result = (nor1 << 1) + nor0;

  return result;
  
}
second_level(a, b, S3, S2, S1, S0, c, m)
     int a;
     int b;
     int S3;
     int S2;
     int S1;
     int S0;
     int c;
     int m;
{
  int c0     = first_level(a & 0x01, b & 0x01, S3, S2, S1, S0);
  int c1     = first_level((a & 0x02) >> 1, (b & 0x02) >> 1, S3, S2, S1, S0);
  int c2     = first_level((a & 0x04) >> 2, (b & 0x04) >> 2, S3, S2, S1, S0);
  int c3     = first_level((a & 0x08) >> 3, (b & 0x08) >> 3, S3, S2, S1, S0);
  
  int carry  = c;
  int mode   = m;

  int c0_0   = (c0 & 0x01);
  int c0_1   = (c0 & 0x02) >> 1;
  int c1_0   = (c1 & 0x01);
  int c1_1   = (c1 & 0x02) >> 1;
  int c2_0   = (c2 & 0x01);
  int c2_1   = (c2 & 0x02) >> 1;
  int c3_0   = (c3 & 0x01);
  int c3_1   = (c3 & 0x02) >> 1;

  int and0   = c3_1;
  int and1   = c3_0 & c2_1;
  int and2   = c3_0 & c2_0 & c1_1;
  int and3   = c3_0 & c2_0 & c1_0 & c0_1;
  int nand0  = ~(c3_0 & c2_0 & c1_0 & c0_0 & carry) & 0x01;
  int nand1  = ~(c3_0 & c2_0 & c1_0 & c0_0) & 0x01;

  int and4   = (carry & 1) & c0_0 & c1_0 & c2_0 & mode;
  int and5   = c1_0 & c2_0 & c0_1 & mode;
  int and6   = c2_0 & c1_1 & mode;
  int and7   = c2_1 & mode;

  int and8   = carry & c0_0 & c1_0 & mode;
  int and9   = c1_0 & c0_1 & mode;
  int and10  = c1_1 & mode;

  int and11  = carry & c0_0 & mode;
  int and12  = c0_1 & mode;

  int nand2  = ~(carry & mode) & 0x01;

  int xor0   = c3_0 ^ c3_1;
  int xor1   = c2_0 ^ c2_1;
  int xor2   = c1_0 ^ c1_1;
  int xor3   = c0_0 ^ c0_1;

  int nor0   = ~(and0 | and1 | and2 | and3) & 0x01;
  int nor1   = ~(and4 | and5 | and6 | and7) & 0x01;
  int nor2   = ~(and8 | and9 | and10) & 0x01;
  int nor3   = ~(and11 | and12) & 0x01;

  int or0    = (~nor0 & 0x01) | (~nand0 & 0x01);
  int xor4   = (xor0 ^ nor1) & 0x01;
  int xor5   = (xor1 ^ nor2) & 0x01;
  int xor6   = (xor2 ^ nor3) & 0x01;
  int xor7   = (xor3 ^ nand2) & 0x01;
  
  int and13  = xor4 & xor5 & xor6 & xor7;
  
  int result = 0;

  result  = nor0;              /* ~G or Y                               */
  result += or0   << 1;        /* Cn+4 or ~Cn+4                         */
  result += nand1 << 2;        /* ~P or X                               */
  result += xor7  << 3;        /* ~F0 or F0                             */
  result += xor6  << 4;        /* ~F1 or F1                             */
  result += xor5  << 5;        /* ~F2 or F2                             */
  result += xor4  << 6;        /* ~F3 or F3                             */
  result += and13 << 7;        /* A = B                                 */

  return result;
}
void
ALUFunctions(A, B, CtlWord)
  int A;
  int B;
  unsigned int CtlWord;
{
  int Alow     = A & 0x0F;
  int Ahigh    = (A & 0xF0) >> 4;
  int Blow     = B & 0x0F;
  int Bhigh    = (B & 0xF0) >> 4;

  int ALU;
  int ALUlow;
  int ALUhigh;

  int Ccompl;
  int E;
  int N;
  int D0; 

  int F3 = (CtlWord & 0x4000) >> 14;
  int F2 = (CtlWord & 0x2000) >> 13;
  int F1 = (CtlWord & 0x1000) >> 12;
  int F0 = (CtlWord & 0x0800) >> 11;
  int C  = (CtlWord & 0x0400) >> 10;
  int M  = ~((CtlWord & 0x0200) >> 9) & 0x01;

  ALUlow  = second_level(Alow , Blow , F3, F2, F1, F0, C, M);
  C       = (ALUlow & 0x02) >> 1;

  ALUhigh = second_level(Ahigh, Bhigh, F3, F2, F1, F0, C, M); 

  ALUReg  = ((ALUlow & 0x78) >> 3) + ((ALUhigh & 0x78) << 1);

  Ccompl = (ALUhigh & 0x02) >> 1;
  E      = ((ALUlow & 0x80) & (ALUhigh & 0x80)) >> 7;
  N      = (ALUhigh & 0x40) >> 6;
  D0     = (ALUlow  & 0x08) >> 3;     

#ifdef USERTLTABLE
  (*LoadFuns[CtlWord&63])();
#endif

  if(CtlWord&0x8000)                            /* bit ADR+                  */
    PCReg=(MicroRomSize-1)&(PCReg+1);
  if( (CtlWord&0x80)==0 )                       /* Teste le bit INHIBIT      */
    { 
    if( CtlWord&0x40 )                          /* Shift (en fait, rotation) */
      {
	CCodeReg=(CCodeReg>>1)+((CCodeReg&1)<<7);
      }
    else                                        /* Load                      */
      {
	CCodeReg = (IOFlag()<<6)        /* Interrupt request de l'UART -> c6 */
	           + Buttons                           /* Poussoirs -> c5,c4 */
	           + (D0  << 3)     /* le bit de poids faible D0 -> c3 */
		   + (E  << 2)                              /* E -> c2 */
		   + (Ccompl << 1)                /* carry out -> c1 */
		   + N;                              /* signe -> c0 */
      }
  }
}

/****************
 * Step-by-step *
 ****************/
  
void NanoStep(unsigned long int maxsteps)
{
/* maxsteps est le nombre de nano-operations apres lequel on arrete d'office */
#ifndef USERTLTABLE
	unsigned int steps=STILLALIVE,tsteps=0,ops5;
	register unsigned int phasereg,CtlWord;
	register byte source;
#endif
	unsigned long int lsteps;
/*	fd = fopen("merde", "w"); */

	RemainingSteps=maxsteps;
	phasereg=PhaseReg;
	ops5=OpReg<<5;
	do
	  {
	    /* Contenu de la rom de controle */
	    CtlWord=CtlRom[(phasereg<<1)+ops5+(CCodeReg&1)];
	    phasereg=(phasereg+1)&0x0F;

	    ALUFunctions(AReg, BReg, CtlWord); /* effectue l'operation de l'ALU pour le 
				      mot de controle courant                */
	  } while( --RemainingSteps );
	
	PhaseReg=phasereg;
	/* Affichage du nombre de pas accomplis */
	lsteps=(unsigned long)tsteps*STILLALIVE+(STILLALIVE-steps);
	if(lsteps>1){
		ultod(lsteps,MsgBuf);
		strcat(MsgBuf," micro-steps completed");
		PutStatus(MsgBuf);
	}else{
		if( lsteps )
			PutStatus("Stepped one micro-instruction");
		else
			PutStatus("Same micro-instruction");
	}
}

void Jump(unsigned int addr ){
	PCTrack=PCReg=addr;
	OpReg=MicroRom[PCReg];
	PhaseReg=0;
}

void Reset(){
	ResetUart();
	CCodeReg=0;
	Jump(0);
}

void InitHard(){
	unsigned long int i;
	SRamTop=&SRam[SRamSize-1];
	DRamTop=&DRam[DRamSize-1];
	MicroRomTop=&MicroRom[MicroRomSize-1];
	CtlRomTop=(byte *)&CtlRom[CtlRomSize-1];
	/* Supposons les Eproms vierges */
#ifdef USEMEMSET
	memset(CtlRom,0xFF,CtlRomSize*2); /* Deux chars par mot */
	memset(MicroRom,0xFF,MicroRomSize);
#else
	for( i=0; i<CtlRomSize; i++) CtlRom[i]=0xFFFF;
	for( i=0; i<MicroRomSize; i++) MicroRom[i]=0xFF;
#endif
	/* Un seul breakpoint par dfaut sur l'instruction bpt=0xFF */
	for( i=0; i<255; i++) BreakOp[i]=0;
	BreakOp[255]=-1;
}
