/*
 * Fichier:mayfile.c
 * Gestion des fichiers dump binaires, rom de contrle et assignations
 */
 
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>

#include "maysim.h"
#include "display.h"
#include "mayfile.h"
#include "hard.h"
#include "status.h"

#define	Comment	(0x7C)
static char Header[TW+1];
char *Headers[256]; /* Un commentaire pour chaque opcode */

/* Envoie un message d'erreur enrichi du numro de ligne */
int LineNum;
void FError(char *ErrMsg){
	strncpy(MsgBuf,ErrMsg,TW-9-5);
	strcat(MsgBuf," at line "); /* 9 chars */
	ultod((long)LineNum,MsgBuf+strlen(MsgBuf)); /* up to 5 chars */
	Error(MsgBuf);
	ErrFlag=TRUE;
}

/* Envoie le message d'erreur correspondant  la lecture
 * d'un caractre illgal
 */
void IllChar(byte c){
        strcpy(MsgBuf,"Illegal character: ascii ");
	ultod((long)c,MsgBuf+strlen(MsgBuf));
	if( c>31 && c<128 ){
	  strcat(MsgBuf,"=' '");
	  MsgBuf[strlen(MsgBuf)-2]=c;
        }
	FError(MsgBuf);
}

/* Lecture d'un nombre binaire de n digits, ventuellement prcd d'espaces
 * Au moins un caractre est toujours lu
 * Il peut y avoir un caractre 'tag' trait comme '0' sauf que l'on
 * gnre un masque permettant de traiter par aprs ses occurences.
 * Le tag peut tre une lettre ou '*'.
 */
static char TagChar;
static unsigned int TagMask,TagCount;

/* Lit un digit, gre le tag */
unsigned int BinDigit(char c){
	if( TagMask&32768 ){
		FError("Too many digits");
	}else{
		TagMask<<=1;	
	}
	switch(c){
		case '1':
			return(1);
		case '0':
			return(0);
		default:
			if( TagChar ){ /* on a dj lu le tag */
				if( c!=TagChar ){
					FError("Only one tag allowed");
					return(-1);
				}
			}else{ /* Nouveau tag */
				if ( !isalpha(c) && c!='*' ){
					FError("Illegal character met");
					return(-1);
				}
				TagChar=c;
			}
			TagCount++; /* Ajuste le nombre d'occurences */
			TagMask++; /* Met le lsb  1, indiquant la position */
			return(0);
	}
}

/* Lecture de n digits binaires (ventuellement prcds d'espaces )
 * L'initialisation de TagMask, TagCount et TagChar (normalement tous nuls,
 * sauf si on lit des champs conscutifs reprsentant une seule valeur)
 * est  charge de la routine appelante.
 */
unsigned int ScanBin(unsigned int n,FILE *f){
	unsigned int i,j;
	char c;
	while( ' '==(c=fgetc(f)) ); /* Saute les blancs, lit le 1er chiffre */
	j=BinDigit(c);
	if( ErrFlag ) return(-1);
	for( i=1;i<n;i++){ /* Lit les n-1 chiffres suivants */
		j=(j<<1)+BinDigit((char)fgetc(f));
		if( ErrFlag ) return(-1);
	}
	return(j);
}

/* Introduits des bits selon un masque
 * Le lsb de val se retrouve  la position du premier bit mis  un dans le
 * masque, le bit de poids 2  celle du second bit mis  un dans le masque etc.
 */
unsigned int SpreadBits(unsigned int val, unsigned int mask){
	unsigned int i=0,outmask=1;
	while( mask && val ){
		while( !(mask&1) ){ /* Cherche le bit suivant du masque */
			mask>>=1; /* Consomme un bit du masque */
			outmask<<=1; /* Ajuste l'emplacement de mise  jour */
		}
		if( val & 1 ) i+=outmask;
		val>>=1; /* Consomme un bit de val */
		mask>>=1;
		outmask<<=1;
	}
	return(i);
}

/* Lit une ligne au format de la rom de contrle */
unsigned long int ReadCtlWord(FILE *f){
	int addr,data,op,vaddr,vdata,vbits;
	unsigned int AddrMask,i,imax,w;
	char c;
	TagChar=0;
	TagMask=0;
	TagCount=0;
	op=ScanBin(8,f); /* Numro d'opcode de microcode */
	addr=op<<5; /* Adresse de base dans la rom de contrle */
	if( ErrFlag ){ FError("Bad op number"); return(0); }
	addr+=ScanBin(4,f)<<1; /* Phase de nanocode */
	if( ErrFlag ){ FError("Bad phase number"); return(0); }
	addr+=ScanBin(1,f); /* Bit de condition */
	if( ErrFlag ){ FError("Bad condition specifier"); return(0); }
	while( ' '==(c=fgetc(f)) ); /* lit le 1er non-blanc */
	if( c!='=' ){ FError("Expected '='"); return(0); }
	/* On connait les bits d'adresse variables */
	AddrMask=TagMask; vbits=TagCount;
	/* A priori, ils ne figurent pas dans les donnes */
	TagMask=0; TagCount=0;
	/* Par contre, le caractre tag reste le mme! */
	data=ScanBin(1,f)<<15; /* Champ ADR+ */
	if( ErrFlag ){ FError("Illegal addr+ specifier"); return(0); }
	data+=ScanBin(4,f)<<11; /* ALU f3..f1 */
	if( ErrFlag ){ FError("Illegal operation specifier"); return(0); }
	data+=ScanBin(2,f)<<9; /* ALU carry in, mode */
	if( ErrFlag ){ FError("Illegal carry/mode specifier"); return(0); }
	data+=ScanBin(1,f)<<8; /* N.C. */
	if( ErrFlag ){ FError("Illegal bit 8 specifier"); return(0); }
	data+=ScanBin(2,f)<<6; /* Contrle de condition */
	if( ErrFlag ){ FError("Illegal condition control"); return(0); }
	data+=ScanBin(3,f)<<3; /* Source */
	if( ErrFlag ){ FError("Illegal source specifier"); return(0); }
	data+=ScanBin(3,f); /* Destination */
	if( ErrFlag ){ FError("Illegal destination specifier"); return(0); }
	if( TagCount>vbits ){
		/* ?? Expansion 1->many si vbits==1 */
		FError("Too many tags on right side");
		return(0);
	}
	imax=1<<vbits; /* 2**vbits combinaisons des bits variables */
	for(i=0;i<imax;i++){
w=data+SpreadBits(i,TagMask);
		CtlRom[addr+SpreadBits(i,AddrMask)]=w;
if  (w&63==63) printf("uart r/w at %u",addr);
	}
	if( Header[0]!=0 && Headers[op]==0 ){ /* Commentaire disponible ? */
		Headers[op]=malloc(strlen(Header)+1);
		if( Headers[op]==NULL ){
			Error("Out of core memory, discarding header");
		}else{
			strcpy(Headers[op],Header);
		}
		Header[0]=0; /* Consomme le header */
	}
	return((long)imax*2); /* 2 bytes par mot! */
}

/* Lit un fichier dcrivant le contenu de la rom de contrle */
unsigned long int ReadCtlRom(FILE *f){
	char c;
	int i;
	boolean GotBlankLine=0;
	unsigned long int totalbytes=0;
	LineNum=1;
	ErrFlag=FALSE;
	while( !feof(f) ){ /* Boucle sur les lignes */
		while( ' '==(c=fgetc(f)) ); /* Saute les blancs */
		switch(c){
			/* ?? possibilit de commencer par un tag */
			case '1':
			case '0':
				ungetc(c,f);
				totalbytes+=ReadCtlWord(f);
				if( ErrFlag ) return(totalbytes);
				GotBlankLine=0;
				break;
			case Comment:
			/* Les commentaires qui suivent une ligne blanche
			 * sont stocks dans 'Header'
			 */
				if( GotBlankLine ){
					GotBlankLine=0;
					i=0;
					while( '\n'!=(c=fgetc(f)) && !feof(f) )
						if( i<TW ) Header[i++]=c;
					Header[i]=0;
				}else{
					while( fgetc(f)!='\n' && !feof(f) );
				}
				LineNum++;
				break;
			/* Permet la lecture de fichiers DOS sans traduction */
			case 0x0D:
				if( fgetc(f)!='\n' ){
				  FError("Unexpected LF");
				  return(totalbytes);
				}
				/* Ligne suivante, pas de break */
			case '\n':
				GotBlankLine=1;
				LineNum++;
				break;
			case EOF:
				break;
			default:
				IllChar((byte)c);
				return(totalbytes);
		}
	}
	return(totalbytes);
}

/**************************************************************
 * Lecture d'un fichier ou d'une portion de fichier dump hexa *
 **************************************************************/

unsigned long int ReadMem(FILE *f, byte *mem, unsigned long int startoffset,
		  unsigned long int endoffset){
	int c,j;
	unsigned int i;
	unsigned long int offset,newoffset,totalbytes=0;
	boolean startline=TRUE,more=TRUE,needetx=FALSE;
	offset=startoffset;
	ErrFlag=TRUE; /* Dpart pessimiste */
	/* STX n'apparaissant licitement qu'une fois, on se dispense
	 * de le tester explicitement  l'intrieur de la boucle.
	 */
	c=fgetc(f);
	if(c==02){ /* On a un STX */
		c=fgetc(f); /* On le saute */
		needetx=TRUE;
	}
	while( more ){ /* Boucle sur les tokens */
		/* Teste d'abord le cas le plus frquent:
		 * Si on a un digit hexa, le token est une valeur de byte
		 * ou une nouvelle adresse.
		 * Notons qu'EOF n'tant pas un token reconnu, il tombe sous
		 * le coup de "Illegal character" et il n'y a pas besoin
		 * de faire un test spar.
		 */
		if( 0<=(j=HexDigit(c)) ){
			c=fgetc(f); /* Deuxime digit, obligatoire */
			i=j;
			if( 0>(j=HexDigit(c)) ){
				FError("Expecting hex digit");
				return(totalbytes);
			}
			i=(i<<4)|j;
			/* Troisime digit ou dbut du token suivant */
			c=fgetc(f);
			if( 0>(j=HexDigit(c)) ){
				/* Pas de troisime digit, on a un byte */
				if( offset<endoffset ){
					totalbytes++;
					mem[offset++]=(byte)i;
				}else{
					FError("Memory boundary exceeded");
					return(totalbytes);
				}
			}else{ /* Troisime digit prsent, on a une adresse */
				newoffset=(long)i;
				/* ?? limiter le nombre de digits */
				do{ /* boucle sur les digits */
					newoffset=(newoffset<<4)|j;
					c=fgetc(f);
					j=HexDigit(c);
				}while( j>=0 );
				if( c==':' ){
					if( newoffset<endoffset ){
						offset=newoffset;	
						c=fgetc(f);
					}else{
						FError("Offset too large");
					}
				}else{
					FError("Expecting ':'");
				}
			}
		/* Le caractre suivant ne peut jamais tre un digit,
		 * pas besoin de boucler.
		 */
		}
		switch(c){ /* Le premier caractre suffit pour reconnatre
			    * les tokens lgaux restants
			    */
		case ' ':
			c=fgetc(f);
			break;
		case '|': /* Absorbe un token "commentaire" */
			/* Le test sur 0x0A permet de lire des fichiers UNIX
			 * sous DOS sans traduction ?? '\n'==0x0a
			 */
			while( '\n'!=(c=fgetc(f)) && 0x0A!=c && !feof(f) );
			/* Ncessairement suivi d'un token n.l., sans break */
		case '\n': /* Absorbe un token "nouvelle ligne" */
			startline=TRUE;
			LineNum++;
			c=fgetc(f);
			break;
		case 0x0D: /* Permet la lecture de fichiers DOS sous UNIX */
			c=fgetc(f);
			if( c!='\n' ){
			  FError("Unexpected LF");
			  return(totalbytes);
			}
			break;
		default:
			more=FALSE;
		}
	}
	if( needetx?(c!=03):(c!=EOF) )
		IllChar((byte)c);
	else
		ErrFlag=FALSE; /* Tout est bien qui finit bien */
	return(totalbytes);
}

/**********************************************
 * Lecture d'un fichier avec noms symboliques *
 **********************************************/
 
/**** pas encore implment mais utilis pour la ligne de commande ****/

/* Les registres prennent des valeurs 8-bits mais sont stocks comme entiers */
boolean SetReg(struct SymbolStr *this, unsigned long int val){
	if(val>255){
		Error("Illegal byte value");
		return(TRUE);
	}
	*(int*)(this->Value)=(int)val;
}

boolean SetPC(struct SymbolStr *this, unsigned long int val){
	if(val>MicroRomSize){
		Error("Illegal microrom address");
		return(TRUE);
	}
	*(word*)(this->Value)=(word)val;
	Jump(PCReg);
}

boolean SetByte(struct SymbolStr *this, unsigned long int val){
	if(val>255){
		Error("Illegal byte value");
		return(TRUE);
	}
	*(byte*)(this->Value)=(byte)val;
}

boolean SetWord(struct SymbolStr *this, unsigned long int val){
	if(val>65535){
		Error("Illegal word value");
		return(TRUE);
	}
	*(word*)(this->Value)=(word)val;
}

boolean SetButton(struct SymbolStr *this, unsigned long int val){
	byte bmask=BUTTONMASK;
	if(val>1){
		Error("Illegal button value");
		return(TRUE);
	}
	if(this->Value) bmask<<=1;
	if(val)
		Buttons|=bmask;
	else
		Buttons&=~bmask;
}

Symbol SymbolTable[]={
	{"A",&AReg,SetReg},
	{"B",&BReg,SetReg},
	{"CCR",&CCodeReg,SetReg},
	{"PC",&PCReg,SetPC},
	{"SW",&Switches,SetByte},
	{"B0",0,SetButton},
	{"B1",&Buttons,SetButton},
	{0,0,0}
};

Symbol *GetSymbol(char *name){
	int i=0;
	char *thisname;
	while(thisname=SymbolTable[i++].Name) /* C'est bien une assignation! */
		if(strcmp(thisname,name)==0) break;
	if(thisname)
		return(&SymbolTable[--i]);
	return(NULL);
}

/***********************************
 * Ecriture d'un fichier dump hexa *
 ***********************************/

unsigned long int WriteMem(FILE *f, byte *mem, unsigned long int startoffset,
		  unsigned long int endoffset, char mode, char *title){
/* mode=0 is raw (for eprogrammer)  */
	unsigned long offset;
	byte *lastbyteaddr,sep;
	sep=*DumpSeparator;
	offset=startoffset;
	lastbyteaddr=&mem[endoffset];
	if(mode==0){
	  fputs("\002\n",f);
	  *DumpSeparator=0;
	}else{
	  fputc('|',f);
	  fputs(title,f);
	  fputc('\n',f);
	  *DumpSeparator='|';
	}
	while( offset<endoffset ){
		if(mode) fprintf(f,"%4.4x: ",offset);
		fputs(SDumpHex(&mem[offset],lastbyteaddr),f);
		fputc('\n',f);
		offset+=16;
	}
	if(mode)
	  fputs("|End of dump\n",f);
	else
	  fputc(03,f);
	*DumpSeparator=sep;
	return(endoffset-startoffset+1);
}

unsigned long int WriteCtlRom(FILE *f, boolean hiflag){
	unsigned offset=0,i,j;
	byte sep,buf[16];
	sep=*DumpSeparator;
	fputs("\002\n",f);
	*DumpSeparator=0;
	while( offset<CtlRomSize ){
		if(hiflag)
			for(i=0;i<16;i++,offset++)
				buf[i]=(byte)(CtlRom[offset]>>8);
		else
			for(i=0;i<16;i++,offset++)
				buf[i]=(byte)CtlRom[offset];
		fputs(SDumpHex(buf,buf+15),f);
		fputc('\n',f);
	}
	fputc(03,f);
	*DumpSeparator=sep;
	return(CtlRomSize);
}

/* Ecriture des registres */
unsigned long int WriteRegs(FILE* f){
	fprintf(f,"\nA=0x%2.2X=%3.3u=%+4.3d\n",AReg,AReg,AReg);
	fprintf(f,"B=0x%2.2X=%3.3u=%+4.3d\n",BReg,BReg,BReg);
	fprintf(f,"ALU result:0x%2.2X=%3.3u=%+4.3d\n",ALUReg,ALUReg,ALUReg);
	fputs("Condition code:",f);
	fputc(CCode&64?' ':'~',f);fputs("Int ",f);
	fputc(CCode&32?' ':'~',f);fputs("P1 ",f);
	fputc(CCode&16?' ':'~',f);fputs("P0 ",f);
	fputc(CCode&8?' ':'~',f);fputs("D0 ",f);
	fputc(CCode&4?' ':'~',f);fputs("E ",f);
	fputc(CCode&2?'~':' ',f);fputs("C ",f);
	fputc(CCode&1?' ':'~',f);fputs("N ",f);
        fprintf(f,"\nCondition code register=%2.2X\n",CCodeReg);
	fprintf(f,"OP=0x%2.2X Phase=0x%1X PC=0x%4.4X\n",OpReg,PhaseReg,PCReg);
	fprintf(f,"Register Address=0x%2.2X->0x%2.2X\n",MAReg,SRam[MAReg]);
	fprintf(f,"Switches=%2.2x\n",Switches);
	return(11);
}

