/*
 * Fichier:maysim.c
 * Fichier principal de l'mulateur MAYBE
 * Voir MAYSIM.DOC pour les notes d'implmentation
 * Voir MAYSIM.MAN pour les notes d'utilisation
 *
 * Version 1.0, 14 fvrier 1993
 */

/* 26/11/1999 (JMF) : Ajout d'une commande Run */

/* #include <stdio.h> */
#include <stdlib.h>
/* Pas de #include <ctype.h> sinon tolower() serait une macro */


#include "maysim.h"
#include "display.h"
#include "maywin.h"
#include "memwin.h"
#include "mayfile.h"
#include "hard.h"
#include "regs.h"
#include "uart.h"
#include "ctlrom.h"
#include "microrom.h"
#include "sram.h"
#include "dram.h"
#include "machine.h"

unsigned long int NiceRead(byte *, unsigned long );

void ParseCommandLine( int, char ** );
int ExitCode=OK;

static char *Banner="Welcome to MAYSIM v.1.0 beta";

boolean BatchMode=TRUE;
boolean WrapMode=TRUE;

unsigned int TermHeight=25;
boolean ForcedHeight=FALSE;

static boolean BadUse=TRUE;
static char *UseMsg="USE: maysim [-option|statement]...\n";
static char *ShortHelpMsg=
"maysim -? will display options summary,"
" see maysim.man for full description.\n";
static char *HelpMsg=
"Available options:\n"
"-?		display this message\n"
"-rwfile	read c|d|m|*r|s contents from file\n"
"-ow		open c|d|m|r|s|u window\n"
/* "		c,d,m and s windows allow @xxxx address sub-option\n" */
/* "		and =nn size setting\n" */
"-ww[file]	write d|m|r|s contents to file or to stdout\n"
"-bnn		set breakpoint on opcode nn\n"
"-g[nn]		go for maximum nn nanosteps\n"
"-i		enter interactive mode even if -g was used\n"
"-hnn		set terminal height to nn\n"
"-w		force wrapping\n"
"where nn is a decimal number or a hex number followed by 'h',\n"
"w is one of the following initials:"
"c:control rom, m:microcode rom, s:static (registers) ram, d:dynamic ram,\n"
"u:uart, command is a register file command.\n"
"Statement is of the form name=nn.\n"
"Beware:-w and -hnn can mess up display and confuse editor when misused.";

/***********************
 * Programme principal *
 ***********************/
main(int argc, char **argv, char **envp){
	puts(Banner);
	/* Initialise AVANT l'interprtation de la ligne de commande! */
	InitHard();
	Reset();
	InitWins();
	ParseCommandLine( argc, argv );

	if( !BatchMode ){
	  InitMachine(); /* Prpare au mode interactif */
	  OpenWin(CreateMicroRomWin());
#if FALSE
	  OpenWin(CreateDRamWin());
	  OpenWin(CreateSRamWin());
#endif
	  OpenWin(CreateCtlRomWin());
	  OpenWin(CreateRegWin());

	  ShowWins();
	  strcpy(MsgBuf,Banner);
	  strcat(MsgBuf,"         ");
	  strcat(MsgBuf,HelpKeyMsg);
	  PutStatus(MsgBuf);
	  EventLoop();
	}

	ExitProgram(ExitCode);
}

void ExitProgram(int exitcode)
{
	ExitMachine();
	if( BadUse ){
		fputs(UseMsg,stderr);
		fputs(ShortHelpMsg,stderr);
	}
	if( exitcode==OK )
		puts("\nThank you for using MAYSIM");
	else
		puts("\nMAYSIM:aborted");
	exit(exitcode);
}

/************************************************************************
 * Gestion des commandes claviers gnrales de l'application		*
 * par opposition au commandes sensibles au contexte (fentre courante) *
 ************************************************************************/
void OpenMenu();
void ReadMenu();
void WriteMenu();
void SetDIL();
void NiceReset();

unsigned int NotifyApp( Event *eventptr ){
	byte brkop;
	unsigned long int steps;
	char **helpline;
	switch(eventptr->Type){
	case KeyPress:
		switch(eventptr->Val){
		case NanoStepKey:
			NanoStep(1);
			Broadcast(&CycleEvent);
			return(OK);
		case StepKey:
			do{ NanoStep(1); } while( PhaseReg ); /* ?terminaison */
			Broadcast(&CycleEvent);
			return(OK);
		case BreakPointKey:
			brkop=GetBVal("Toggle breakpoint for opcode #?",OpReg);
			if (!ErrFlag){
				BreakOp[brkop]^=-1;
				PutStatus(BreakOp[brkop]?
					"Breakpoint set":"Breakpoint cleared");	
			}
			return(OK);
		case GoKey:
		        steps=GetVal("Go for how many nano-steps?");
			if( !ErrFlag ){
			  PutStatus("Running...");
			  NanoStep(steps);
			  Broadcast(&CycleEvent);
			}
			return(OK);
		case RunKey:  /* (JMF) */
		  PutStatus("Running a lot of nanosteps...");
		  NanoStep(RUNSTEPS);
		  Broadcast(&CycleEvent);
		  return(OK);
		case OpenKey:
			OpenMenu();
			return(OK);
		case ReadKey:
			ReadMenu();
			return(OK);
		case WriteKey:
			WriteMenu();
			return(OK);
		case ResetKey:
			NiceReset();
			return(OK);
		case Button0Key:
			Buttons^=BUTTONMASK;
			Broadcast(&RegEvent);
			return(OK);
		case Button1Key:
			Buttons^=BUTTONMASK<<1;
			Broadcast(&RegEvent);
			return(OK);
		case DILSwitchKey:
			SetDIL();
			return(OK);
		case GHelpKey: /* ?? Type de fentre propre */
			GotoXY(0,0);
			Title("On-line help",0);
			helpline=HelpText;
			while(*helpline[0]){
				putsi(*helpline++);
				NextWinLine();
			}
			Title("End of help",0);
			MarkAll();
			Warn("See also file maysim.man");
			return(OK);
		}
		break;
	}
	return(Dispatch(eventptr));
}

/* Sous-menu d'ouverture d'une fentre */
int OpenSelect( int key ){
	switch(tolower(key)){
		case 's':
			OpenWin(CreateSRamWin());
			break;
		case 'd':
			OpenWin(CreateDRamWin());
			break;
		case 'c':
			OpenWin(CreateCtlRomWin());
			break;
		case 'r':
			OpenWin(CreateRegWin());
			break;
		case 'm':
			OpenWin(CreateMicroRomWin());
			break;
		case 'u':
			OpenWin(CreateUartWin());
			break;
		default:
			return(FAIL);
	}
	return(OK);
}

void OpenMenu(){
	if( FreeWinLines()<2 ){ /* Pas de place pour ouvrir */
		Warn(NoRoomMsg);
		return;
	}
	PutStatus("Open window: Sram Dram Ctlrom Microrom Regs Uart");
	if( OpenSelect(GetKey())==OK )
		PutStatus(IdleStatus);
	else
		PutStatus(CancelStatus);
}

/* Lecture d'un fichier mmoire avec affichage interactif */
unsigned long int NiceRead(byte *mem, unsigned long endoffset){
	FILE *f;
	unsigned long int t=0;
	/* ?? Demande le startoffset */
	if( NULL!=(f=OpenFile("r","Read from file:")) ){
		PutStatus("Reading...");
		LineNum=1;
		t=ReadMem(f,mem,0,endoffset);
		PutReadStatus(t);
	}
	return(t);
}

/* Sous-menu de lecture d'un fichier */
void ReadMenu(){
	FILE *f;
	Event event0;
	unsigned long int t;
	PutStatus("Read from file: Sram Dram Ctlrom Microrom All");
	switch(tolower(GetKey())){
	case 's':
		if(NiceRead(SRam,SRamSize)>0) Broadcast(&SRamEvent);
		break;
	case 'd':
		if(NiceRead(DRam,DRamSize)>0) Broadcast(&DRamEvent);
		break;
	case 'c':
		if( NULL!=(f=OpenFile("r","Read control rom file:")) ){
			t=ReadCtlRom(f);
			PutReadStatus(t);
			if( t>0 ){
				event0.Type=CtlRomUpdate;
				Broadcast(&event0);
			}
		}
		break;
	case 'm':
		if(NiceRead(MicroRom,MicroRomSize)>0){
			event0.Type=MicroRomUpdate;
			Broadcast(&event0);
			/* Le byte sur lequel pointe PC peut avoir chang */
			/* On suppose que PC pointe sur un micro-opcode. */
			Jump(PCReg);
			Broadcast(&PCEvent);
		}
		break;
/*	case 'a': read all pas encore crit */
	default:
		PutStatus(CancelStatus);
	}
}

/* Ecriture d'un fichier mmoire avec affichage interactif */
void PutWriteStatus(unsigned long int t){
	strcpy(MsgBuf,"Wrote ");
	ultod(t,MsgBuf+strlen(MsgBuf));
	strcat(MsgBuf," bytes.");
	PutStatus(MsgBuf);
}

unsigned long int NiceWrite(byte *mem, unsigned long maxoffset,
			    char mode, char *title){
	FILE *f;
	unsigned long int t=0, startoffset, endoffset;
	switch( mode ){
	case 'h': /* Format humain, dump partiel possible */
		do{
		  startoffset=GetDVal("Start offset [0]?",0);
		  if( ErrFlag ) return(0);
		}while(startoffset>maxoffset);
		do{
		  endoffset=GetDVal("End offset [Top of memory]?",maxoffset);
		  if( ErrFlag ) return(0);
		}while(endoffset>maxoffset && startoffset>endoffset);
		break;
	case 0: /* Format eprogrammeur, toujours du dbut  la fin */
		startoffset=0;
		endoffset=maxoffset;
		break;
	default:
		return(0);
	}
	if( NULL!=(f=OpenFile("w","Write to file:")) ){
		PutStatus("Writing...");
		LineNum=1;
		t=WriteMem(f, mem, startoffset, endoffset, mode, title);
		fclose(f);
		PutWriteStatus(t);
	}
	return(t);
}

/* Pour la rom microcode, deux modes d'criture sont reconnus:
 * 0->eprogrammeur, 'h'->humain
 */
char GetWriteMode(){
  char c;
  PutStatus("Write format:(H)uman, (E)programmer");
  while( 'h'!=(c=tolower(GetKey())) && 'e'!=c && AbortKey!=c);
  if( c==AbortKey ) return(-1);
  if( c=='e' ) c=0;
  return(c);
}

/* Sous-menu d'criture d'un fichier */
void WriteMenu(){
	FILE *f;
	Event event0;
	unsigned long int t;
	PutStatus("Write to file: Sram Dram Ctlrom Microrom All");
	switch(tolower(GetKey())){
	case 's':
		NiceWrite(SRam,SRamSize-1,'h',"MAYBE Static ram dump");
		break;
	case 'd':
		NiceWrite(DRam,DRamSize-1,'h',"MAYBE dynamic ram dump");
		break;
	case 'c':
		/* NiceWrite((byte *)CtlRom,CtlRomSize*2-1,
			  GetWriteMode(), "MAYBE control rom dump");*/
		f=OpenFile("w","Control rom (lower) file:");
		if( f!=NULL ) WriteCtlRom(f,FALSE);
		f=OpenFile("w","Control rom (upper) file:");
		if( f!=NULL ) WriteCtlRom(f,TRUE);
		break;
	case 'm':
		NiceWrite(MicroRom,MicroRomSize-1,
			  GetWriteMode(), "MAYBE microcode rom dump");
		break;
/*	case 'a': Write all pas encore crit */
	default:
		PutStatus(CancelStatus);
	}
}

void SetDIL(){
	char c;
	PutStatus("Toggle DIL switch (0..7)?");
	c=GetKey();
	if( c<'0' || c>'7' ){
		PutStatus(CancelStatus);
	}else{
		Switches^=1<<(c-'0');
		Broadcast(&RegEvent);
		PutStatus(IdleStatus);
	}
}

void NiceReset(){
	PutStatus("Reset MAYBE (y/n)?");
	if(tolower(GetKey())=='y'){
		Reset();
		Broadcast(&PCEvent);
		PutStatus("Reset!");
	}else{
		PutStatus(CancelStatus);
	}
}

/******************************************
 * Interprtation de la ligne de commande *
 ******************************************/

/* Vrifie l'absence d'autre caractres aprs une option */
void LoneOption(char * option){
	if( option[2] ){
		fprintf(stderr,"Option -%c must be used by itself\n",
			option[1]);
		ExitProgram(FAIL);
	}
}
void WinError(char *option){
	fprintf(stderr,"Illegal window specifier in option %s\n",option);
	ExitProgram(FAIL);
}

int ReadSelect(char, char *);
int WriteSelect(char, char *);

void ParseCommandLine( int argc, char **argv ){
	FILE *f;
	Symbol *symptr;
	char *thisarg;
	unsigned int i,j;
	unsigned long int t;
	boolean forcebatchmode=FALSE,forceinteractivemode=FALSE;
	/* Test des diffrentes options */
	for(i=1;i<argc;i++){
		thisarg=argv[i];
		if(*thisarg=='-'){ /* Est-ce une option? */
			switch(tolower(thisarg[1])){
			case '?': /* Aide */
				puts(UseMsg);
				puts(HelpMsg);
				ExitProgram(OK);
			case 'r': /* Lecture mmoire */
				if( thisarg[2]==0 || thisarg[3]==0 ){
					fputs("Missing read specifier",stderr);
					ExitProgram(FAIL);
				}
				if( ReadSelect(thisarg[2],thisarg+3)!=OK )
					WinError(thisarg);
				break;
			case 'o': /* Ouverture d'une fentre */
				j=2;
				while( thisarg[j]!=0 )
					if( OpenSelect(thisarg[j++])!=OK )
						WinError(thisarg);
				forceinteractivemode=TRUE;
				break;
			case 'b': /* Force un breakpoint sur opcode */
				t=Eval(thisarg+2);
				if(ErrFlag || t>255){
					fputs("Out of range breakpoint\n",
						stderr);
					ExitProgram(FAIL);
				}
				BreakOp[t]=-1;
				break;
			case 'i': /* Force le mode interactif */
				LoneOption(thisarg);
				forceinteractivemode=TRUE;
				break;
			case 'g': /* Go ventuellement suivi d'une limite */
				if( thisarg[2] ){
					t=Eval(thisarg+2);
					if( ErrFlag ){
						fputs("Invalid 'Go' limit\n",
							stderr);
						ExitProgram(FAIL);
					}
				}else{
					t=500000; /* Limite par dfaut */
				}
				fputs("Running",stdout);
				BadUse=FALSE;
				UpdateALU();
				NanoStep(t);
				BadUse=TRUE;
				forcebatchmode=TRUE;
				break;
			case 'h': /* Hauteur d'cran */
				TermHeight=Eval(thisarg+2);
				if( ErrFlag ){
					fputs("Invalid height\n",stderr);
					ExitProgram(FAIL);
				}
				ForcedHeight=TRUE;
				break;			
			case 'w':
				if( thisarg[2] ){ /* Ecriture mmoire */
					if( WriteSelect(thisarg[2],thisarg+3)
						!=OK )
						WinError(thisarg);
					break;
				}else{ /* Wrap (CR aprs toute les lignes) */
					WrapMode=FALSE;
				}
				break;
			default:
				fprintf(stderr,"%s: Unknown option %s\n",
					argv[0],thisarg);
		       		ExitProgram(FAIL);
			}
		}else{ /* Assignation symbolique */
			j=0;
			while( thisarg[j] && thisarg[j]!='=' ) j++;
			if( thisarg[j]!='=' ){
				fputs("Syntax error\n",stderr);
		       		ExitProgram(FAIL);
			}
			BadUse=FALSE;
			thisarg[j++]=0;
			symptr=GetSymbol(thisarg);
			if( symptr==NULL ){
				fprintf(stderr,"Unknown symbol:%s\n", thisarg);
				ExitProgram(FAIL);
			}
			t=Eval(&thisarg[j]);
			if( ErrFlag ){
				fprintf(stderr,"Illegal value:%s\n",
					thisarg+j);
				ExitProgram(FAIL);
			}
			symptr->SetVal(symptr,t);
			BadUse=TRUE;
		}
	}
	/* On ne dcide de passe ventuellement en mode interactif
	 * qu'une fois l'examen des options termin.
	 */
	BatchMode=forceinteractivemode?FALSE:forcebatchmode;
	BadUse=FALSE;
	/* UpdateALU(); Certaines options ont pu la modifier */
}

/* Traitement de l'option -r */
int ReadSelect(char c, char *filename){
	FILE *f;
	unsigned long int t;
	BadUse=FALSE; /* Les fautes de lecture ne sont pas des fautes d'usage */
	f=fopen(filename,"r");
	if( f==NULL ){
		fprintf(stderr,"Cannot open %s\n",filename);
		ExitProgram(FAIL);
	}
	LineNum=0;
	switch(tolower(c)){
	case 's':
		fputs("Loading static ram...",stdout);
		t=ReadMem(f,SRam,0,SRamSize);
		break;
	case 'd':
		fputs("Loading dynamic ram...",stdout);
		t=ReadMem(f,DRam,0,DRamSize);
		break;
	case 'm':
		fputs("Loading microcode rom...",stdout);
		t=ReadMem(f,MicroRom,0,MicroRomSize);
		if( t>0 )
			/* Le byte sur lequel pointe PC peut avoir chang */
			Jump(PCReg);
		break;
	case 'c':
		fputs("Loading control rom...",stdout);
		t=ReadCtlRom(f);
		break;
	default: /* Erreur dans l'option */
		fclose(f);
		BadUse=TRUE;
		return(FAIL);
	}
	if( ErrFlag ){
		fprintf(stderr,"Error reading file %s\n",filename);
			ExitProgram(FAIL);
	}
	PutReadStatus(t);
	BadUse=TRUE;
	return(OK);
}

/* Traitement de l'option -w */
int WriteSelect(char c, char *filename){
	FILE *f;
	unsigned long int t;
	BadUse=FALSE;
	if( *filename ){
		f=fopen(filename,"w");
		if( f==NULL ){
			fprintf(stderr,"Cannot open %s\n",filename);
		ExitProgram(FAIL);
		}
	}else{
		f=stdout;
	}
	switch(tolower(c)){
	case 's':
		fputs("Writing static ram...",stdout);
		t=WriteMem(f,SRam,0,SRamSize,'h',"Static ram contents");
		break;
	case 'd':
		fputs("Writing dynamic ram...",stdout);
		t=WriteMem(f,DRam,0,DRamSize,'h',"Dynamic ram contents");
		break;
	case 'm':
		fputs("Writing microcode rom...",stdout);
		t=WriteMem(f,MicroRom,0,MicroRomSize,
			'h',"Microcode rom contents");
		break;
/*	case 'c':
		fputs("Writing control rom...",stdout);
		t=WriteCtlRom(f);
		break;
*/
	case 'r':
		fputs("Writing registers...",stdout);
		t=WriteRegs(f);
		break;
	default: /* Erreur dans l'option */
		fclose(f);
		BadUse=TRUE;
		return(FAIL);
	}
	if( ErrFlag ){
		fprintf(stderr,"Error writing file %s\n",filename);
			ExitProgram(FAIL);
	}
	PutWriteStatus(t);
	BadUse=TRUE;
	return(OK);
}
