/*
 * Fichier:maywin.c
 * Gestion du fentrage horizontal pour l'mulateur MAYBE
 */

#include <string.h>
/* #include <stdio.h> */
#include <stdlib.h>
/* Pas de #include <ctype.h> sinon tolower() serait une macro */

#include "maysim.h"
#include "display.h"
#include "maywin.h"
#include "machine.h"

static void ShowWin(Win*);
static void ShrinkWin(Win*);
static void EnlargeWin(Win*);
static unsigned int GetY(Win*);
static void * GetWin(unsigned int);

/********************************************
 * Gestion du tableau des fentres visibles *
 ********************************************/
unsigned int CurX=0,CurY=0; /* Position du curseur */
static unsigned int LastLine;

/* Tableau des fentres affichables */
static Win *Visible[MAXWINS],*Zoomed[MAXWINS];
static boolean IsZoomed=FALSE;
static unsigned int ZoomHeight,ZoomX,ZoomY;

/* Initialisation gnrale du fentrage */
void InitWins(){
	LastLine=TermHeight-StatusHeight;
}


/* Hauteur occupe  l'affichage */
unsigned int TotalWinLines(){
	unsigned int i,j=0;
	for(i=0;i<MAXWINS;i++)
		if (Visible[i]) j+=TitleHeight+Visible[i]->Height;
	return(j);
}
/* Dernire fentre visible */
Win *LastWin(){
	unsigned int i;
	if(!Visible[0]) return(0);
	for(i=1; Visible[i] && i++<MAXWINS;);
	return(Visible[i-1]);
}

/* Fentre possdant la ligne Row */
static void * GetWin(unsigned int Row){
	Win *WinPtr;
	unsigned int i,j=0;
	for(i=0;i<MAXWINS;i++){
		WinPtr=Visible[i];
		if(WinPtr){
			j+=TitleHeight+WinPtr->Height;
			if( j>Row ) return(WinPtr);
/* printfi("%2u",WinClassPtr->MinHeight); */
		}
	}
	return(0);
}

/* Ligne relative  la fentre courante */
/* Pour cette fonction, le bas de l'cran est considr comme appartenant
 *  la dernire fentre, mme si celle-ci est en fait plus courte.
 * 0 signifie que le curseur se trouve sur la ligne de titre de la fentre.
 */
int RelY(){
	int i=0,top=0,oldtop=0;
	while( top<CurY && i<MAXWINS ){
		if( Visible[i] ){
			oldtop=top;
			top+=TitleHeight+Visible[i]->Height;
		}
		i++;
	}
	return(CurY-oldtop);
}

void * CheckWin(unsigned int Row){
	Win *WinPtr;
	WinPtr=GetWin(Row);
	if (WinPtr==0){
		Error("Cursor must be on valid window");
	}
	return(WinPtr);
}

/* Marque les fentres qui suivent *WinPtr comme sales */
void MarkFrom(Win *WinPtr){
	int i;
	for( i=0; i<MAXWINS && Visible[i]!=WinPtr; i++ );
	for( ; i<MAXWINS; i++ )
		if( Visible[i] ) Visible[i]->Dirty=TRUE;
}
void MarkAll(){
	MarkFrom(Visible[0]);
	LastLine=TermHeight-2; /* Numero de la ligne avant celle de statut */
}

/* Ouvre une fentre
 * Toute fentre doit tre cre avant qu'on puisse l'afficher.
 */
char *NoRoomMsg="No room for opening - close or shrink other window";
void OpenWin(Win *WinPtr){
	unsigned int i,j,k;
	int f;
	Win *wptr;
	if( IsZoomed ){
	  Error("May not open while in zoom mode");
	  return;
	}
	if( WinPtr ){
		f=FreeWinLines()-TitleHeight;
		if( WinPtr->Height>f
		    && WinPtr->WinClass0->MinHeight<=f)
		    	WinPtr->Height=f;
		if( WinPtr->Height<=f)
			for(i=0;i<MAXWINS;i++)
				if( Visible[i]==0 ){
					wptr=GetWin( CurY );
					/* Boucle passe si wptr=0 */
					while( Visible[i]!=wptr ){
						Visible[i]=Visible[i-1];
						Visible[i]->Dirty=TRUE;
						i--;
					}
					WinPtr->Dirty=TRUE;
					Visible[i]=WinPtr;
					return;
				}
		Error(NoRoomMsg);
		WinPtr->Height=0;
	}else{
		Error("Cannot open window");
	}
}

/* Retourne le nombre de lignes blanches  l'cran si le nombre
 * maximum de fentres n'est pas atteint, son oppos sinon.
 */
int FreeWinLines(){
	unsigned int i,j=StatusHeight;
	boolean foundfree=FALSE;
	for(i=0; i<MAXWINS; i++)
		if(Visible[i]){
			j+=TitleHeight+Visible[i]->Height;
		}else{
			foundfree=TRUE;
		}
	j=TermHeight-j;
	return(foundfree?j:-j);
}

/* Commute le mode plein cran sans vrification - usage interne */
void ZoomIn(Win *winptr){
	unsigned int i;
	for(i=0; i<MAXWINS; i++){
		Zoomed[i]=Visible[i];
		Visible[i]=0;
	}
	ZoomHeight=winptr->Height;
	ZoomX=CurX;
	ZoomY=CurY;
	Visible[0]=winptr;
	i=TermHeight-TitleHeight-StatusHeight;
	if(i>winptr->WinClass0->MaxHeight)
		i=winptr->WinClass0->MaxHeight;
	winptr->Height=i;
	winptr->Dirty=TRUE;
	IsZoomed=TRUE;
}
void ZoomOut(){
	unsigned int i;
	Visible[0]->Height=ZoomHeight;
	for(i=0; i<MAXWINS; i++)
		Visible[i]=Zoomed[i];
	MarkAll();
	CurY=ZoomY;
	CurX=ZoomX;
	IsZoomed=FALSE;
}

/* Ferme une fentre */
void CloseWin(Win *WinPtr){
	unsigned int i;
	if( IsZoomed ) ZoomOut();
	if( WinPtr ){
		MarkFrom(WinPtr);
		for(i=0;i<MAXWINS;i++){
			if( Visible[i]==WinPtr ){
				WinPtr->Height=0;
				while(++i<MAXWINS) Visible[i-1]=Visible[i];
				Visible[MAXWINS-1]=0;
				return;
			}
		}
		Error("Cannot close window");
	}
}

/* Rtrcit une fentre */
static void ShrinkWin(Win *WinPtr){
	if( WinPtr ){
		if( WinPtr->Height>WinPtr->WinClass0->MinHeight ){
			--(WinPtr->Height);
			if( GetWin(CurY)!=WinPtr )
				CurY--;
				MarkFrom(WinPtr);
		}else{
			Error("Cannot shrink window further");
		}
	}
}

/* Agrandit une fentre */
static void EnlargeWin(Win *WinPtr){
	if( WinPtr ){
		if( WinPtr->Height<WinPtr->WinClass0->MaxHeight
		    && FreeWinLines()>0 ){
			++(WinPtr->Height);
			MarkFrom(WinPtr);
		}else{
			Error("Cannot enlarge window further");
		}
	}
}

/* Affiche une fentre */
static void ShowWin(Win *WinPtr){
	unsigned int y;
	y=GetY(WinPtr);
	if (y==FAIL){
		Error("Cannot show window");
	}else{
		GotoXY(0,y);
		WinPtr->WinClass0->Display(WinPtr);
		WinPtr->Dirty=FALSE;
	}
}

/* Affiche toutes les fentres visibles */
void ShowWins(){
	WinClass *WinClassPtr;
	Win *WinPtr;
	unsigned int i,j=0;
	for(i=0;i<MAXWINS;i++){
		WinPtr=Visible[i];
		if(WinPtr){
			if( WinPtr->Dirty){
				/* Le GotoXY garantit l'alignement ds la
				 * premire fentre affiche.
				 */
				GotoXY(0,j);
				WinPtr->WinClass0->Display(WinPtr);
				WinPtr->Dirty=FALSE;
			}
			j+=TitleHeight+WinPtr->Height;
		}
	}
	i=LastLine;LastLine=j;
	if( j<i ){ /* La dernire ligne a remont */
		GotoXY(0,j); /* Dbut de la zone  nettoyer */
		while( j++<i ) NextWinLine();
	}
	/* Affiche la liste des hauteurs ou des classes pour le debugging */
#if FALSE
	for(i=0;i<MAXWINS;i++){
		WinPtr=Visible[i];
		if(WinPtr)
			printfi("%2.2u ",WinPtr->Height);
/*			printfi("%p ",WinPtr->WinClass0);*/
		else
			putsi("-- ");
	}
	printfi("Total height:%2u",TotalWinLines());
#endif
}

/* A quelle ligne commence la fentre WinPtr? */
static unsigned int GetY(Win *WinPtr){
	unsigned int i,j=0;
	Win *ThisWin;
	for(i=0;i<MAXWINS;i++){
		ThisWin=Visible[i];
		if (ThisWin){
			if(ThisWin==WinPtr)
				return(j);
			else
				j+=TitleHeight+ThisWin->Height;
		}
	}
	return(FAIL);
}

/**************
 * Evnements *
 **************/
 
/* Evnements prdfinis pour conomiser une initialisation  chaque usage */
Event PCEvent={PCUpdate};
Event OpcodeEvent={OpcodeUpdate};
Event PhaseEvent={PhaseUpdate};
Event RegEvent={RegUpdate};
Event SRamEvent={SRamUpdate};
Event DRamEvent={DRamUpdate};
/* CycleEvent rsume tous les prcdents et doit tre reconnu par toutes les
 * fentres qui reconnaissent l'un d'eux.
 */
Event CycleEvent={Cycle};

/* Signale un vnement  toutes les fentres visibles */
void Broadcast(Event *event0){
	unsigned int i;
	Win *winptr;
	for(i=0;i<MAXWINS;i++)
		if(winptr=Visible[i])
			winptr->WinClass0->Notify(winptr,event0);
}

/* Signale un vnement clavier  la fentre du curseur */
unsigned int Dispatch( Event *eventptr ){
	Win *winptr;
	winptr=GetWin(CurY);
	if (winptr)
		return(winptr->WinClass0->Notify(winptr,eventptr));
	else
		return(FAIL);
}

/* Ragit  un vnement concernant le fentrage */
int NotifyWins(Event *eventptr){
	Win *winptr;
	unsigned int h;
	switch(eventptr->Type){
	case KeyPress:
		switch(eventptr->Val){
		case UpKey:
			CurY--;
			break;
		case DownKey:
			CurY++;
			break;
		case LeftKey:
			CurX--;
			break;
		case RightKey:
			CurX++;
			break;
		case HomeKey:
			if( CurX ){
				CurX=0;
			}else{
				winptr=GetWin(CurY);
				if( winptr ){
					h=GetY(winptr);
/* +winptr->WinClass0->MinHeight-1; */
					if( CurY>h+TitleHeight ){
						CurY=h+TitleHeight;
					}else{
						winptr=GetWin(h-1);
						if( winptr )
							CurY=GetY(winptr)
							     +TitleHeight;
/* +winptr->WinClass0->MinHeight-1; */
					}
				}else{
					CurY=0;
				}
			}
			break;
		case EndKey:
			winptr=GetWin(CurY);
			if( winptr ){
				if (CurY<(h
				  =GetY(winptr)+TitleHeight+winptr->Height-1) )
					CurY=h;
				else
					if( winptr=GetWin(CurY+1) )
						CurY=GetY(winptr)+TitleHeight
						    +winptr->Height-1;
			}
			break;
		case CloseKey:
			CloseWin(CheckWin(CurY));
			break;
		case RefreshKey:
			MarkAll();
			break;
		case ZoomKey:
			if(IsZoomed){
				ZoomOut();
				break;
			}
			winptr=GetWin(CurY);
			if (winptr==0) winptr=LastWin();
			if (winptr){
				if( FreeWinLines()==0 && 
				  winptr->WinClass0->MaxHeight>winptr->Height){
				  	ZoomIn(winptr);
					break;
				}
				h=winptr->Height+abs(FreeWinLines());
				/* +TermHeight-1-TotalWinLines(); */
				if(h>winptr->WinClass0->MaxHeight)
					h=winptr->WinClass0->MaxHeight;
				winptr->Height=h;
				MarkFrom(winptr);
			}else{
				Error("Cursor must be on valid window");
			}
			break;
		case EnlargeKey:
			EnlargeWin(CheckWin(CurY));
			break;
		case ShrinkKey:
			ShrinkWin(CheckWin(CurY));
			break;
		default:
			return(FAIL);
		}
		if( CurX>=TW )
			CurX=TW-1;
		if( CurY>=TermHeight-StatusHeight)
			CurY=TermHeight-StatusHeight-1;
		break;
	default:
		return(FAIL);
	}
	return(OK);
}
void UnknownKey(unsigned int key){
  strcpy(MsgBuf,"Unknown key:");
  ultod((unsigned long)key,MsgBuf+strlen(MsgBuf));
  strcat(MsgBuf,"        ");
  strcat(MsgBuf,HelpKeyMsg);
  Error(MsgBuf);
}
/* Attente des vnements */
static Event KeyEvent={KeyPress};
void EventLoop(){
	unsigned int Key;
	while ((Key=GetKey())!=QuitKey){
		PutStatus(IdleStatus);
		KeyEvent.Val=Key;
		if( NotifyWins(&KeyEvent)==FAIL
		    && NotifyApp(&KeyEvent)==FAIL )
		  UnknownKey(Key);
		ShowWins();
		GotoXY(CurX,CurY);
	}
	PutStatus("");
	GotoXY(0,TermHeight-1);
}

/*********************************
 * Gestion de la ligne de statut *
 *********************************/
 
char *IdleStatus=">";
char *CancelStatus="Cancelled";
void PutStatus(char *StatusString){
	unsigned int i;
	if( BatchMode ){
		putchar('\n');
		puts(StatusString);
	}else{
		GotoXY(0,TermHeight-1);
		putsi(StatusString);
		ClEol();
		GotoXY(CurX,CurY);
/*		Refresh();*/
	}
}

/* Ecrit un message justifie a droite dans la ligne de statut */
void PutRStatus(char *msg){
/* La longueur de msg doit etre inferieure a TW-2 */
  if(BatchMode){
    puts(msg);
  }else{
    /* Un bug de curses ou dans xterm fait qu'il ne faut pas se positionner
     * dans le coin inferieur droit de la fenetre sous peine de planter
     * getch() d'ou le -2
     */
    GotoXY(TW-strlen(msg)-2,TermHeight-1);
    putsi(msg);
  }
}

/* Attend que l'utilisateur ait lu l'affichage */
void Pause(){
	if(BatchMode) return;
        PutRStatus("Press Enter");
        while(GetKey()!=EnterKey);
	GotoXY(TW-12,TermHeight-1);
	ClEol();
}

/* Affiche un message et attend que l'utilisateur l'ait lu */
void Warn( char *WarnMsg){
	PutStatus(WarnMsg);
	if(BatchMode) return;
	Pause();
	PutStatus(IdleStatus);
}

/* Affiche un message d'erreur, sort du programme si necessaire */
void Error( char *ErrMsg){
	PutStatus(ErrMsg);
	Bell();
	if(BatchMode) ExitProgram(FAIL);
	Pause();
	PutStatus(IdleStatus);
}

/* Lit une ligne avec possibilit d'dition */
char *GetString( char *prompt ){
	static char answer[TW]; /* ?? scrolling pour lignes >TW */
	unsigned int c,i=0,imax,j,plen;
	boolean dirty=FALSE;
	answer[0]=0;
	ErrFlag=FALSE;
	GotoXY(0,TermHeight-1);
	putsi(prompt);
	ClEol();
	plen=strlen(prompt);
	imax=TW-plen-1;
	while( EnterKey!=(c=GetKey()) && i<imax ){
		switch( c ){
		case BSKey:
			if( i>0 ){
				 strcpy(&answer[i-1],&answer[i]);
				 i--;
				 dirty=TRUE;
			}
			break;
		case DelKey:
			if( i<(j=strlen(answer))){
				 strcpy(&answer[i],&answer[i+1]);
				 dirty=TRUE;
			}
			break;
		case LeftKey: 
			if( i>0 ) i--;
			break;
		case HomeKey:
			i=0;
			break;
		case RightKey:
			if( i<strlen(answer) ) i++;
			break;
		case EndKey:
			i=strlen(answer);
			break;
		case KillKey:
			answer[i]=0;
			ClEol();
			break;
		case AbortKey:
			ErrFlag=TRUE;
			PutStatus("Aborted!");
			return(answer);
		case QuoteKey: /* Ctrl-q force l'insertion d'un caractre */
			c=GetKey(); /* ? Affichage chars de contrle */
			/* pas de break, on insre */
		default:
			if( c>31 && c<128 ){
				/* Le test en tte de boucle garantit
				 * qu'il y a place pour insrer
				 */
				for(j=1+strlen(answer); j>i; j--)
					answer[j]=answer[j-1];
				answer[i]=(char)c;
				putsi(&answer[i]);
				/* Vu que le string s'allonge, pas
				 * besoin de ClEol()
				 */
				i++;
			}else{
				Bell();
			}
		}
		if( dirty ){
			dirty=FALSE;
			GotoXY(plen,TermHeight-1);
			putsi(answer);
			ClEol();
		}
		GotoXY(plen+i,TermHeight-1);
	}
	PutStatus(IdleStatus);
	return(answer);
}

/* Ouverture interactive d'un fichier
 * ErrFlag est positionn si l'utilisateur annulle (par Ctrl-G) mais pas si
 * le fichier ne peut tre ouvert. Par contre, le rsultat est NULL dans les
 * deux cas.
 */
FILE *OpenFile(char *access, char *msg){
	FILE *f;
	char *filename;
	filename=GetString(msg);
	if( ErrFlag ) return(NULL);
	f=fopen(filename,access);
	if( f==NULL ) Error("Open failed");
	return(f);
}

boolean ErrFlag;

/* Affiche un message appropri aprs une lecture fichier */
void PutReadStatus( unsigned long int totalbytes ){
	strcat( ultod(totalbytes,MsgBuf), " bytes read, " );
	if(ErrFlag)
		strcat(MsgBuf,"abnormal read termination");
	else
		strcat(MsgBuf,"no error encountered");
	PutStatus(MsgBuf);
}

char *StripBl(char *line){
	while( *line==' ' ) line++;
	return(line);
}

/* Evalue une chane de caractres, met  jour ErrFlag
 * La chane doit reprsenter un entier non sign, dcimal, hexa ou binaire.
 * La chane peut tre vide ou prcde de blancs.
 * Un nombre trs grand (>2^32-1) peut tre tronqu.
 */
unsigned long int Eval(char *digits){
	int d,i=0,l;
	unsigned long int v=0;
	char *sptr;
	ErrFlag=FALSE;
	sptr=StripBl(digits);
	l=strlen(sptr);
	if( l ){
		switch( tolower(sptr[l-1]) ){
		case 'h':
			while( (d=HexDigit(sptr[i++]))>=0 ) v=(v<<4)+d;
			if( i==l ) return(v); /* Arrt sur le 'h' */
			break;
		case 'b':
			while( (d=sptr[i++]-'0')>=0 && d<=1 ) v=(v<<1)+d;
			if( i==l ) return(v); /* Arrt sur le 'b' */
			break;
		default: /* Ceci ressemble  atoi mais dtecte les erreurs */
			while( (d=(sptr[i++]-'0'))>=0 && d<=9 ) v=10*v+d;
			if( i>l ) return(v); /* Arrt sur le 0 final */
		}
	}
	ErrFlag=TRUE;
	return(0);
}

/* Lit une valeur numrique */
unsigned long int GetVal(char *prompt){
	char *sptr;
	sptr=GetString(prompt);
	if( ErrFlag ) return(0);
	return(Eval(sptr));
}

/* Lit une valeur numrique, accepte une valeur par dfaut */
unsigned long int GetDVal(char *prompt, unsigned long int defval){
	unsigned long int d;
	char *sptr;
	sptr=(GetString(prompt));
	if( !ErrFlag && *sptr ){
		d=Eval(sptr);
		if(!ErrFlag) return(d);
	}
	return(defval);
}

/* Lit une valeur de byte, accepte une valeur par dfaut */
byte GetBVal(char *prompt, byte defval){
	unsigned long int v;
	v=GetDVal(prompt,(unsigned long int) defval);
	if( v>255 ){
		Error("Illegal byte value");
		ErrFlag=TRUE;
		return(defval);
	}
	return((byte)v);
}
