/*----------------------------------------------------------------
 * This module is the master piece of the flash
 * programmer/simulator device.
 *
 * mapping of the ppi bits :
 *  pc0 = A16
 *  pc1 = /ce
 *  pc2 = /rd
 *  pc3 = /wr
 *  pc4 = led_on
 *  pc5 = 0:prog/1:sim
 *  pc6 = clock for xa8..15 register (74374)
 *  pc7 = 1:activate the target /reset line.
 *----------------------------------------------------------------
 */

#include "ppi_io.h"

#include <stdio.h>

#ifdef __TURBOC__
#include <conio.h>
#include <alloc.h>
#else
#include <malloc.h>
#endif

#include <string.h>
#include <errno.h>

#ifdef __TURBOC__
#include <io.h>
#include <sys\stat.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif

#include <fcntl.h>

#define WRITING 1
#define READING 2
#define INFO 3
#define GO 4
#define STOP 5

#define FBUF_SIZE 5000

unsigned char pcCache = 0;
/* used to avoid a read from PORT_C, in order to speed
   up the download */

unsigned char fbuf[FBUF_SIZE];

/*----------------------------------------------------------------
 * flashWrite writes a byte into the flash memory
 * INPUT : add is the address to write and value the data to
 * be written
 * OUTPUT : nothing.
 *----------------------------------------------------------------
 */
void flashWrite(unsigned long add, unsigned char val) {
  unsigned char addLow, addHigh, add16;

  addLow = (unsigned char) (add & 0xff);
  addHigh = (unsigned char) ((add >> 8) & 0xff);
  add16 = (unsigned char) ((add >> 16) & 0x1);
  pcCache = (pcCache & 0xfe) | add16; /* prepare a16 */

  ppiWrite(PORT_B, addLow);
  ppiWrite(PORT_A, addHigh);

  /* write xa8..15 and a16 */
  /* pc6 assumed to be low... */
  pcCache |= 0x4e;
  ppiWrite(PORT_C, pcCache); /* rise pc6, /ce, /rd and /wr */
  pcCache &= 0xbd;
  ppiWrite(PORT_C, pcCache); /* lower pc6 *AND* /ce */

  /* write access */
  ppiWrite(PORT_A, val);
  ppiWrite(PORT_C, pcCache & 0xf7); /* lower /wr */
  ppiWrite(PORT_C, pcCache | 0x0a); /* rise /wr and /ce */
}

/*----------------------------------------------------------------
 * flashRead read a byte from the flash
 * INPUT : add is the address to read from
 * OUTPUT : read byte
 *----------------------------------------------------------------
 */
unsigned char flashRead(unsigned long add) {
  unsigned char addLow, addHigh, add16, v;

  addLow = (unsigned char) (add & 0xff);
  addHigh = (unsigned char) ((add >> 8) & 0xff);
  add16 = (unsigned char) ((add >> 16) & 1);
  pcCache = (pcCache & 0xfe) | add16; /* prepare a16 */

  ppiWrite(PORT_B, addLow);
  ppiWrite(PORT_A, addHigh);

  /* write xa8..15 and a16 */
  /* pc6 assumed to be low... */
  pcCache |= 0x4e;
  ppiWrite(PORT_C, pcCache); /* rise pc6, /rd, /wr and /ce */
  pcCache &= 0xbd;
  ppiWrite(PORT_C, pcCache); /* lower pc6 *AND* /ce */

  /* revert port A direction */
  ppiWrite(CONTROL_WORD, 0x90); /* a input b, c output */
  ppiWrite(PORT_C, pcCache); /* lower pc6 *AND* /ce */
  ppiWrite(PORT_B, addLow);

  ppiWrite(PORT_C, pcCache & 0xfb); /* lower /rd */
  v = ppiRead(PORT_A); /* read the byte */
  ppiWrite(PORT_C, pcCache | 0x04); /* rise /rd */

  ppiWrite(CONTROL_WORD, 0x80); /* all ports for output */
  ppiWrite(PORT_C, pcCache | 0x06); /* rise /rd and /ce */

  return v;
}

/*----------------------------------------------------------------
 * ledOn turns the led on the flash card on
 *----------------------------------------------------------------
 */
void ledOn() {
  pcCache |= 0x10;
  ppiWrite(PORT_C, pcCache);
}

/*----------------------------------------------------------------
 * ledOff turns the led off (ha ha!)
 *----------------------------------------------------------------
 */
void ledOff() {
  pcCache &= 0xef;
  ppiWrite(PORT_C, pcCache);
}

/*----------------------------------------------------------------
 * run let start the application connected to the flash
 *----------------------------------------------------------------
 */
void run() {
  pcCache &= 0x7f;
  ppiWrite(PORT_C, pcCache);
}

/*----------------------------------------------------------------
 * reset resets the application and maintains a reset condition
 *----------------------------------------------------------------
 */
void reset() {
  pcCache |= 0x80;
  ppiWrite(PORT_C, pcCache);
}

/*----------------------------------------------------------------
 * here are the functions to implement the basic flash commands
 *----------------------------------------------------------------
 */

void flashReset() {
  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0xf0);
}

unsigned char flashManID() {
  unsigned char r;
  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0x90);
  r = flashRead(0x0);
  flashReset();
  return r;
}

unsigned char flashDevID() {
  unsigned char r;
  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0x90);
  r = flashRead(0x1);
  flashReset();
  return r;
}

/*----------------------------------------------------------------
 * flashProgram returns -1 if it fails and 0 otherwise...
 *----------------------------------------------------------------
 */
int flashProgram(unsigned long add, unsigned char val) {
  int done;
  unsigned char a;

  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0xa0);
  flashWrite(add, val);
  done = 0;
  while(!done) {
    a = flashRead(add);
    if ((a & 0x80) == (val & 0x80)) done = 1;
    else if (a & 0x20) {
      a = flashRead(add);
      if ((a & 0x80) == (val & 0x80)) done = 1;
      else done = 2;
    }
  }
  if (done == 2) return (-1); else return 0;
}

/*----------------------------------------------------------------
 * flashEraseChip returns -1 if it fails and 0 otherwise...
 *----------------------------------------------------------------
 */
int flashEraseChip() {
  int done;
  unsigned char a;

  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0x80);
  flashWrite(0x5555, 0xaa);
  flashWrite(0x2aaa, 0x55);
  flashWrite(0x5555, 0x10);
  done = 0;
  while(!done) {
    a = flashRead(0x0);
    if ((a & 0x80) == 0x80) done = 1;
    else if (a & 0x20) {
      a = flashRead(0x0);
      if ((a & 0x80) == 0x80) done = 1;
      else done = 2;
    }
  }
  if (done == 2) return (-1); else return 0;
}


void usage(char *progname) {
  printf("Usage : %s -w <filename> [-sNNNNNN] : programs the " \
	 "flash starting \n\t\tat NNNNNN (0 as default).\n\n" \
	 "\t%s -r <filename> [-sNNNNNN] [-lMMMMMM] : reads " \
	 "MMMMMM bytes\n\t\t(128Kb as default) starting " \
	 "at NNNNNN (0 as default),\n\n" \
	 "\t%s -g : starts the application.\n\n" \
	 "\t%s -s : stops the application (reset).\n\n" \
	 "\t%s -i : displays infos " \
	 "about the flash.\n\n" \
	 "\t0 <= NNNNNN < 131072 ; 0 <= MMMMMM <= 131072.\n",
	 progname, progname, progname, progname, progname);
}

/*----------------------------------------------------------------
 *----------------------------------------------------------------
 */
int main(int argc, char** argv) {
  int i, a, mode, fd, curByte, rd;
  long fsize = 0;
  unsigned long curAdd, u;
#ifndef __TURBOC__
  char *progname = strrchr(argv[0],'/') + 1;
#else
  char *progname = strrchr(argv[0],'\\') + 1;
#endif

	
  if (argv[1][0] != '-') {
    usage(progname);
    exit(-1);
  }
  if (argc < 2) {
    usage(progname);
    exit(-1);
  }

  a = (int) argv[1][1];
  switch(a) {
  case 'w' :
    if (argc == 3) {
      curAdd = 0;
    }
    else if ((argc == 4) && (argv[3][0] == '-')
	     && (argv[3][1] == 's')) {
      if(sscanf(argv[3]+2, "%lu", &curAdd) != 1) {
	usage(progname);
	exit(-1);
      }
      if (curAdd > 131071) {
	usage(progname);
	exit(-1);
      }
    }
    else {
      usage(progname);
      exit(-1);
    }

    printf("%s: start address %lu assumed.\n", progname, curAdd);
    mode = WRITING;
    break;

  case 'r' :
    curAdd = 0;
    fsize = 1024L * 128;

    if (argc < 3) {
      usage(progname);
      exit(-1);
    }

    for(i = 3; i < argc; i++) {
      if (argv[i][0] != '-') {
	usage(progname);
	exit(-1);
      }
      a = (int) argv[i][1];
      switch(a) {

      case 's':

	if(sscanf(argv[i]+2, "%lu", &curAdd) != 1) {
	  usage(progname);
	  exit(-1);
	}
	if (curAdd > 131071) {
	  usage(progname);
	  exit(-1);
	}
	break;

      case 'l':
	if(sscanf(argv[i]+2, "%lu", &u) != 1) {
	  usage(progname);
	  exit(-1);
	}
	if ((u < 0) || (u > 131072)) {
	  usage(progname);
	  exit(-1);
	}
	fsize = (long) u;
	break;

      default:
	usage(progname);
	exit(-1);
      }
    }
    printf("%s: start address %lu assumed, reading " \
	   "%lu byte(s)\n",
	   progname, curAdd, (unsigned long) fsize);

    if((curAdd + fsize) > 131072) {
      printf("%s: NNNNNN + MMMMMM > 131072 (128Kb)!\n",
	     progname);
      usage(progname);
      exit(-1);
    }
    mode = READING;
    break;

  case 'i' :
    if (argc != 2) {
      usage(progname);
      exit(-1);
    }
    mode = INFO;
    break;
  case 'g' :
    if (argc != 2) {
      usage(progname);
      exit(-1);
    }
    mode = GO;
    break;
  case 's' :
    if (argc != 2) {
      usage(progname);
      exit(-1);
    }
    mode = STOP;
    break;
  case '?' :
  case 'h' :
    usage(progname);
    exit(0);
  default :
    usage(progname);
    exit(-1);
  }

  if ((i = ppiOpen(AUTO_PROBE)) != -1)
    printf("%s: PPI interface found on port 0x%x.\n",
	   progname, i);
  else {
    printf("%s: PPI interface not found.\n", progname);
    exit(-1);
  }
  ppiWrite(CONTROL_WORD, 0x80); /* all ports are outputs */

  pcCache = 0xae;
  ppiWrite(PORT_C, pcCache); /* see bof for mapping */
  ppiWrite(PORT_A, 0x0);
  ppiWrite(PORT_B, 0x0);

  switch(mode) {

  case READING :

    printf("%s: reading...", progname);
    fflush(NULL);

    /* create the file */
#ifdef __TURBOC__
    if ((fd = _creat(argv[2], S_IFREG)) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
#else
    if ((fd = creat(argv[2], S_IFREG)) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
#endif
    /* read the flash (128Kb) */
    while(fsize) {
      rd = FBUF_SIZE < fsize ? FBUF_SIZE : fsize;
      fsize -= rd;
      curByte = 0;
      pcCache & 0x10 ? ledOff() : ledOn();
      while(curByte < rd) {
	fbuf[curByte++] = flashRead(curAdd++);
      }
      if (write(fd, fbuf, rd) == -1) {
	perror(progname);
	ppiClose();
	exit(-1);
      }
    }
    if (close(fd) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
    printf("\b\b\b complete.\n");
    ledOff();
    break;

  case WRITING :

    /* open the file and get its size */
#ifdef __TURBOC__
    if ((fd = open(argv[2], O_RDONLY |
		   O_BINARY)) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
#else
    if ((fd = open(argv[2], O_RDONLY)) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
#endif
    if ((fsize = (long) lseek(fd, 0, SEEK_END)) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
    if (lseek(fd, 0, SEEK_SET) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }

    if((curAdd + fsize) > 131072) {
      printf("%s: NNNNNN + MMMMMM > 131072 (128Kb)!\n",
	     progname);
      usage(progname);
      ppiClose();
      exit(-1);
    }

    /* erase the flash */
    if (flashEraseChip() == -1) {
      printf("%s: flash can not be erased!\n", progname);
      ppiClose();
      exit(-1);
    }
    else printf("%s: flash erased...\n", progname);

    /* write it to the flash */
    printf("%s: writing...\n", progname);
    while(fsize) {
      rd = FBUF_SIZE < fsize ? FBUF_SIZE : fsize;
      fsize -= rd;
      if (read(fd, fbuf, rd) == -1) {
	perror(progname);
	ppiClose();
	exit(-1);
      }
      curByte = 0;
      pcCache & 0x10 ? ledOff() : ledOn();
      while(curByte < rd) {
	if (flashProgram(curAdd++, fbuf[curByte++]) == -1) {
	  printf("%s: flash write error\n", progname);
	  ppiClose();
	  exit(-1);
	}
      }
    }
    if (close(fd) == -1) {
      perror(progname);
      ppiClose();
      exit(-1);
    }
    printf("%s: programming complete.\n", progname);
    ledOn();
    pcCache &= 0xdf; ppiWrite(PORT_C, pcCache);
    run();
    break;

  case INFO :
    printf("%s: manufacturer ID = 0x%x, device ID = 0x%x.\n",
	   progname, flashManID(), flashDevID());
    break;

  case STOP :
    ledOff();
    reset();
    break;

  case GO :
    reset();
    ledOn();
    pcCache &= 0xdf;
    ppiWrite(PORT_C, pcCache); /* flash seen from cpu bus */
    run();
    break;
  }

  if (ppiClose() == -1)
    printf("%s: PPI could not be closed...\n", progname);
  else
    printf("%s: PPI interface closed.\n", progname);

  return 0;

}
