#include "capteur.h"
#include "lowlevel/ppi_io.h"
#include <stdio.h>  /* printf */
#include <stdlib.h> /* exit */
#include "processes.h" /* semaphore */
#include <errno.h>

static unsigned char led_status; /* port B leds status */
static int semid;

/*--------------------------------------------------------------*/
int cap_init(void) {
  int i;

  if ((i = ppiOpen(AUTO_PROBE)) != -1)
    printf("PPI interface found on parallel port base %#x.\n",i);
  else {
    printf("PPI interface not found.\n");
    return -1;
  }
  ppiWrite(CONTROL_WORD, 0x90);
  ppiWrite(PORT_B, 0); /* all LED's off. */

  /* semaphore creation */
  if((semid = sema_create(1)) == -1) {
    printf("sema_create() error\n");
    exit(errno);
  }

  /* semaphore initialization */
  if(sema_set_value(semid, 0, 1) == -1){
    printf("sema_set_value() error\n");
    exit(errno);
  };
  return 0;
}

/*--------------------------------------------------------------*/
int cap_self_check(void) {
  return 0;
}

/*--------------------------------------------------------------*/
int cap_close(void) {

  sema_free(semid); /* free semaphore */

  if (ppiClose() == -1) {
    printf("PPI could not be closed...\n");
    return -1;
  }
  else {
    printf("PPI interface closed.\n");
    return 0;
  }
}

/*--------------------------------------------------------------*/
int cap_test_captor(int num) {
  unsigned char bitnum, in; 

  if ((num > NB_CAPTORS) || (num < 1)) {
    printf("Captor number out of range!\n");
    exit(-1);
  }
  
  bitnum = 1 << (--num);
  sema_wait(semid, 0);
  in = ppiRead(PORT_A) & bitnum;
  sema_signal(semid, 0);  
  return (int) !in;
}

/*--------------------------------------------------------------*/
void cap_select_camera(int num) {
  if ((num != CAP_CAMERA_BALL) && (num != CAP_CAMERA_BEACON)) {
    printf("Camera number out of range!\n");
    exit(-1);
  }

  sema_wait(semid, 0);
  ppiWrite(PORT_C, ~((unsigned char) --num) & 3);
  sema_signal(semid, 0);
}

/*--------------------------------------------------------------*/
void cap_led_on(int num) {
  led_status |= (1 << --num);
  sema_wait(semid, 0);
  ppiWrite(PORT_B, led_status);
  sema_signal(semid, 0);
}

/*--------------------------------------------------------------*/
void cap_led_off(int num) {
  led_status &= ~(1 << --num);
  sema_wait(semid, 0);
  ppiWrite(PORT_B, led_status);
  sema_signal(semid, 0);
}

