/*----------------------------------------------------------------
 * This module is the interface between the application and the
 * ppi interface. For now, it uses the lp_io routines to access
 * this hardware (through the parallel interface...), but in the
 * future, it will access it by using a dedicated ppi module and
 * ioctl commands. A mechanism to allow irq's is also missing.
 * This will be added later (!!! cache irq enable bit in order
 * to avoid the need of a (slow) read to the command register !!!)
 *
 * // (base+1) register layout (status)
 * I N N N N - - -      I : inverted N : normal
 * 7 6 5 4 3 2 1 0
 * _ _     _            PPI interface :
 * b a p s e 0 0 0         /buzy  = D3/7
 * u c e l r               pe     = D2/6
 * z k   c r               slct   = D1/5
 * y     t o               /error = D0/4
 *         r               /ack   = /int (unused for now)
 *
 *  // (base+2) register layout (command)
 * X X X N I N I I      I : inverted N : normal
 * 7 6 5 4 3 2 1 0
 *           _   _      PPI interface :
 *       i s i a s         /init     = /read
 *       r l n u t         selct in  = /write
 *       q c i t r         /strobe   = a0
 *         t t o o         auto feed = a1
 *       e       b
 *       n i   f e
 *       a n   e
 *       b     e
 *       l     d
 *       e
 *----------------------------------------------------------------
 */

#include "ppi_io.h"
#include "lookup.c"
#include <stdio.h>

static int ppiProbe(void);

ppi_struct ppiStruct[PPI_MAX] = {
  {0x3bc, 4, 0},
  {0x378, 4, 0},
  {0x278, 4, 0}
};

/*----------------------------------------------------------------
 * ppiWrite writes a byte to the 8255 ppi.
 *
 * INPUT : 'ppi_port' is the port address (0..3) where 'value'
 * has to be written to.
 *
 * OUTPUT : nothing.
 *----------------------------------------------------------------
 */
void ppiWrite(unsigned short ppi_port, unsigned char value) {

  unsigned char c;

  /* write data */
  lpWriteData(value);

  /* it seems that if the 8255 is in read mode (i.e. the /read
     line is still low), the write sequence occurs properly but
     if /read is high and the address to be written changes at
     the same moment /write goes low, it sucks a lot; so the next
     two lines are necessary! */
  c = ((ppi_port ^ 3) & 3) | 0x04;
  lpWriteCommand(c);

  /* write A1 and A0 *AND* lower /WR */
  c = ((ppi_port ^ 3) & 3) | 0x0c;
  lpWriteCommand(c);

  /* rise /WR */
  c = c & 0xf7;
  lpWriteCommand(c);
}

/*----------------------------------------------------------------
 * ppiRead reads a byte from the 8255 ppi.
 *
 * INPUT : 'ppi_port' is the port address (0..3) from which the
 * byte will be read.
 *
 * OUTPUT : the byte value.
 *----------------------------------------------------------------
 */
unsigned char ppiRead(unsigned short ppi_port) {

  unsigned char c, v;

  /* select least significant nibble */
  lpWriteData(0);

  /* write A1 and A0 *AND* lower /RD */
  c = (ppi_port ^ 3) & 3;
  lpWriteCommand(c);

  /* read nibble */
  v = lsnTbl[lpReadStatus()];

  /* select most significant nibble */
  lpWriteData(1);

  /* read nibble */
  v |= msnTbl[lpReadStatus()];

  /* rise /RD */
  c = c | 4;
  lpWriteCommand(c);

  return v;
}

/*----------------------------------------------------------------
 * ppiOpen must be called before any call to another routine of
 * this module.
 *
 * INPUT : 'num' is the number of the parallel interface to be
 * used. The correct values to be used are defined in 'ppi_io.h' :
 * AUTO_PROBE to scan the ports and try to find a ppi interface.
 * LP0 for the first printer port (0x3bc)
 * LP1 for the second printer port (0x378)
 * LP2 for the third printer port (0x278).
 *
 * OUTPUT : selected port address if successfull and -1 if not.
 *----------------------------------------------------------------
 */
int ppiOpen(int num) {
  int i;

  switch(num) {

  case AUTO_PROBE : 

    for(i = 0; i < PPI_MAX; i++) {
      if(lpSelectPort(ppiStruct[i].base, ppiStruct[i].extent)) {
	perror("lpSelectPort()");
	return (-1);
      }
      if (ppiProbe() == 0) {
	ppiStruct[i].selected = 1;
	return (ppiStruct[i].base);
      }
    }
    return (-1);

  case LP0 :
  case LP1 :
  case LP2 :
    if(lpSelectPort(ppiStruct[num].base, ppiStruct[num].extent)) {
      perror("lpSelectPort()");
      return (-1);
    }
    if (ppiProbe() == -1) return (-1);
    ppiStruct[num].selected = 1;
    return (ppiStruct[num].base);

  default : 
    return (-1);
  }
}

/*----------------------------------------------------------------
 * ppiClose should be called when operations with the ppi are
 * finished.
 *
 * INPUT : nothing.
 *
 * OUTPUT : 0 if successfull and -1 if not.
 *----------------------------------------------------------------
 */
int ppiClose(void) {
  int i;

  for(i = 0; i < PPI_MAX; i++) {
    if (ppiStruct[i].selected) {
      ppiStruct[i].selected = 0;
      lpReleasePort();
      return 0;
    }
  }
  return (-1);
}

/*----------------------------------------------------------------
 * ppiProbe is an internal function that tries to find a ppi
 * interface connected to the current selected port.
 *
 * INPUT : nothing.
 *
 * OUTPUT : 0 if a ppi has been found on the port and -1 if not.
 *----------------------------------------------------------------
 */
static int ppiProbe(void) {
  unsigned char c;

  /* rise /RD to activate the 245 (this will have
     no effect on a printer connected to the port) */
  c = (lpReadCommand() & 0xfb) | 4;
  lpWriteCommand(c);

  lpWriteData(0x5a);
  if (lsnTbl[lpReadStatus()] != 0x0a) return (-1);
  lpWriteData(0xa4);
  if (lsnTbl[lpReadStatus()] != 0x04) return (-1);
  lpWriteData(0x5b);
  if (msnTbl[lpReadStatus()] != 0x50) return (-1);
  lpWriteData(0xa5);
  if (msnTbl[lpReadStatus()] != 0xa0) return (-1);
  return 0;
}
