/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009, 2010, 2011 Sébastien PIERARD
 */

// TODO : handle errors in JPG ios
// For informations about libjpeg, see :
//   * http://refspecs.freestandards.org/LSB_3.1.1/LSB-Desktop-generic/LSB-Desktop-generic/libjpegman.html
//   * http://www.ijg.org/

#include "GrayscaleImage.h"

#include <cstdio>
#include <stdexcept>
#include <iostream>

#include <setjmp.h>
#include <jpeglib.h>
#include <jerror.h>

#include <string.h>

GrayscaleImage :: GrayscaleImage ( unsigned int width , unsigned int height ) {
	this -> width = width ;
	this -> height = height ;
	this -> stride = width ;
	this -> pixels = new unsigned char [ stride * height ] ;
	memset ( pixels , 0 , stride * height * sizeof ( unsigned char ) ) ;
}

void GrayscaleImage :: saveAsPGM ( const char * path ) {

	if ( path == NULL ) throw std::invalid_argument ( "" ) ;
	FILE * stream = fopen ( path , "w" ) ;
	if ( stream == NULL ) throw std::runtime_error ( "" ) ;
	fprintf ( stream , "P5\n%d %d\n255\n" , width , height ) ;
	for ( unsigned int row = 0 ; row < height ; ++ row ) {
		fwrite ( pixels + row * stride , sizeof ( unsigned char ) , width , stream ) ;
	}
	fclose ( stream ) ;

}

void GrayscaleImage :: saveAsJPG ( const char * path , int quality ) {

	if ( quality < 0 || quality > 100 ) std::invalid_argument ( "JPEG quality must be between 0 and 100" ) ;

	FILE * stream = fopen ( path , "wb" ) ;
	if ( stream == NULL ) {
		throw std::runtime_error ( "Unable to open file" ) ;
	}

	struct jpeg_error_mgr err ;
	struct jpeg_compress_struct cinfo ;
	cinfo.err = jpeg_std_error ( & err ) ;

	jpeg_create_compress ( & cinfo ) ;
	jpeg_stdio_dest ( & cinfo , stream ) ;
	cinfo.image_width = width ;
	cinfo.image_height = height ;
	cinfo.input_components = 1 ;
	cinfo.in_color_space = JCS_GRAYSCALE ;
	jpeg_set_defaults ( & cinfo ) ;
	jpeg_set_quality ( & cinfo , quality , TRUE ) ;
	jpeg_start_compress ( & cinfo , TRUE ) ;
	while ( cinfo.next_scanline < height ) {
		JSAMPROW row_pointer = & pixels [ cinfo.next_scanline * stride ] ;
		jpeg_write_scanlines ( & cinfo , & row_pointer , 1 ) ;
	}
	jpeg_finish_compress ( & cinfo ) ;
	jpeg_destroy_compress ( & cinfo ) ;

	fclose ( stream ) ;

}

GrayscaleImage :: GrayscaleImage ( const char * path ) {

	FILE * stream = fopen ( path , "rb" ) ;
	if ( stream == NULL ) {
		throw std::runtime_error ( "Unable to open file" ) ;
	}

	struct jpeg_error_mgr err ;
	struct jpeg_decompress_struct cinfo ;
	cinfo.err = jpeg_std_error ( & err ) ;

	// read headers

	jpeg_create_decompress ( & cinfo ) ;
	jpeg_stdio_src ( & cinfo , stream ) ;
	jpeg_read_header ( & cinfo , TRUE ) ;
	width = cinfo.image_width ;
	height = cinfo.image_height ;
	stride = width ;
	pixels = new unsigned char [ stride * height ] ;

	// decompress

	cinfo.output_width = width ;
	cinfo.output_height = height ;
	cinfo.out_color_space = JCS_GRAYSCALE ;
	cinfo.output_components = 1 ;
	jpeg_start_decompress ( & cinfo ) ;
	JSAMPARRAY buffer = ( * cinfo.mem -> alloc_sarray ) ( ( j_common_ptr ) & cinfo , JPOOL_IMAGE , stride , 1 ) ;
	long i=0;
	while ( cinfo.output_scanline < height ) {
		jpeg_read_scanlines ( & cinfo , buffer , 1 ) ;
		memcpy ( pixels + i , buffer [ 0 ] , stride ) ;
		i += stride ;
	}

	jpeg_finish_decompress ( & cinfo ) ;
	jpeg_destroy_decompress ( & cinfo ) ;
	fclose ( stream ) ;

}

GrayscaleImage :: ~ GrayscaleImage () {
	delete [] pixels ;
}
