/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009, 2010, 2011 Sébastien PIERARD
 */

#include <iostream>
#include <stdlib.h>
#include <vector>
#include <algorithm>
#include <stdexcept>
#include <fstream>

#include "Granu.h"
#include "Volume2D.h"
#include "Rectangle.h"
#include "GrayscaleImage.h"
#include "DefaultVolume2D.h"
#include "RectangleMeasure.h"

using namespace std ;

static void usage ( char * command ) {
	cout << "USAGE : " << command << " <input-jpg-file> <filter> <output-m-file>" << endl ;
	cout << "        where <filter> should be WIDTH, HEIGHT, AREA, DIAGONAL, CENTER-X, or CENTER-Y" << endl ;
	exit ( EXIT_FAILURE ) ;
}

class RectangleSorter {
private :
	RectangleMeasure * measure ;
public :
	RectangleSorter ( RectangleMeasure * measure ) : measure ( measure ) {}
	bool operator () ( Rectangle r1 , Rectangle r2 ) const {
		const RectangleMeasure & ref = * measure ;
		return ref ( r1 ) < ref ( r2 ) ;
	}
	~ RectangleSorter () {}
} ;

int main ( int argc , char * argv [] ) {

	{ // this block is needed to avoid memory leak when calling exit
		// because otherwise RAII would be unused.

		if ( argc != 4 ) usage ( argv [ 0 ] ) ;
		RectangleMeasure * measure_ptr = NULL ;
		try { measure_ptr = RectangleMeasure :: getMeasureByName ( argv [ 2 ] ) ; }
		catch ( ... ) { usage ( argv [ 0 ] ) ; }
		const RectangleMeasure & measure_ref = * measure_ptr ;

		cout << "********************************************************************************" << endl ;
		cout << "*  IMPORTANT NOTE. This program was written with the primary objective of the  *" << endl ;
		cout << "*  source code readability. It has not been optimized. Thus, this program      *" << endl ;
		cout << "*  should not be used to measure the performance of algorithms.                *" << endl ;
		cout << "********************************************************************************" << endl ;

		cout << "Loading image ..." << endl ;

		GrayscaleImage img ( argv [ 1 ] ) ;
		Volume2D * volume = Volume2D :: createVolumeFromImage ( img ) ;

		cout << "Listing largest included rectangles ..." << endl ;

		vector < Rectangle > listing ;
		Granu :: addLargestIncludedRectangles ( * volume , listing ) ;
		const int num_rectangles = listing.size () ;
		cout << "INFO : number of rectangles = " << num_rectangles << endl ;

		if ( num_rectangles > 0 ) {

			cout << "Sorting rectangles by " << argv [ 2 ] << " ..." << endl ;

			sort ( listing.begin () , listing.end () , RectangleSorter ( measure_ptr ) ) ;

			cout << "Computing granulometries ..." << endl ;

			AxisAlignedBoundingBox aabb = volume -> getBoundingBox () ;
			DefaultVolume2D reconstructed_lt ( aabb.getMinX () , aabb.getMaxX () , aabb.getMinY () , aabb.getMaxY () ) ;
			DefaultVolume2D reconstructed_gt ( aabb.getMinX () , aabb.getMaxX () , aabb.getMinY () , aabb.getMaxY () ) ;

			double * const threshold_values = new double [ num_rectangles ] ;
			for ( int i = 0 ; i < num_rectangles ; ++ i ) {
				threshold_values [ i ] = measure_ref ( listing [ i ] ) ;
			}
			double * const reconstructed_areas_lt = new double [ num_rectangles ] ;
			for ( int i = 0 ; i < num_rectangles ; ++ i ) {
				reconstructed_lt.add ( listing [ i ] ) ;
				reconstructed_areas_lt [ i ] = reconstructed_lt.getArea () ;
			}
			double * const reconstructed_areas_gt = new double [ num_rectangles ] ;
			for ( int i = num_rectangles - 1 ; i >= 0 ; -- i ) {
				reconstructed_gt.add ( listing [ i ] ) ;
				reconstructed_areas_gt [ i ] = reconstructed_gt.getArea () ;
			}

			cout << "Writing matlab/octave script to plot results ..." << endl ;

			ofstream output_matlab_script ;
			output_matlab_script.open ( argv [ 3 ] , ios::out ) ;
			output_matlab_script << "% This file has been machine generated, do not edit !" << endl ;
			output_matlab_script << "% ( this is a script compatible with Matlab and Octave )" << endl ;
			output_matlab_script << "% Command used to produce this file :" << endl ;
			output_matlab_script << "%     " << argv [ 0 ] << " " << argv [ 1 ] << " " << argv [ 2 ] << " " << argv [ 3 ] << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "threshold = [ ..." << endl ;
			for ( int i = 0 ; i < num_rectangles ; ++ i ) {
				output_matlab_script << "\t" << threshold_values [ i ] << " , ..." << endl ;
			}
			output_matlab_script << "] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "threshold = reshape ( repmat ( threshold , 2 , 1 ) , [ 1 " << ( 2 * num_rectangles ) << " ] ) ; " << endl ;
			output_matlab_script << "threshold = [ -inf threshold inf ] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "area_lt = [ ..." << endl ;
			for ( int i = 0 ; i < num_rectangles ; ++ i ) {
				output_matlab_script << "\t" << reconstructed_areas_lt [ i ] << " , ..." << endl ;
			}
			output_matlab_script << "] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "area_lt = reshape ( repmat ( area_lt , 2 , 1 ) , [ 1 " << ( 2 * num_rectangles ) << " ] ) ; " << endl ;
			output_matlab_script << "area_lt = [ 0 0 area_lt ] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "area_gt = [ ..." << endl ;
			for ( int i = 0 ; i < num_rectangles ; ++ i ) {
				output_matlab_script << "\t" << reconstructed_areas_gt [ i ] << " , ..." << endl ;
			}
			output_matlab_script << "] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "plot_lt  = yes_or_no ( 'Do you want to plot the curve for < ? ' ) ;" << endl ;
			output_matlab_script << "plot_gt  = yes_or_no ( 'Do you want to plot the curve for > ? ' ) ;" << endl ;
			output_matlab_script << "plot_sum = yes_or_no ( 'Do you want to plot the sum ? ' ) ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "area_gt = reshape ( repmat ( area_gt , 2 , 1 ) , [ 1 " << ( 2 * num_rectangles ) << " ] ) ; " << endl ;
			output_matlab_script << "area_gt = [ area_gt 0 0 ] ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script << "figure ;" << endl ;
			output_matlab_script << "hold on ;" << endl ;
			output_matlab_script << "curve_names = {} ;" << endl ;
			output_matlab_script << "if ( plot_lt ) ;" << endl ;
			output_matlab_script << "\tplot ( threshold , area_lt , 'b' ) ;" << endl ;
			output_matlab_script << "\tcurve_names ( length ( curve_names ) + 1 ) = ' select rectangles such that value < threshold ' ;" << endl ;
			output_matlab_script << "end ;" << endl ;
			output_matlab_script << "if ( plot_gt ) ;" << endl ;
			output_matlab_script << "\tplot ( threshold , area_gt , 'r' ) ;" << endl ;
			output_matlab_script << "\tcurve_names ( length ( curve_names ) + 1 ) = ' select rectangles such that value > threshold ' ;" << endl ;
			output_matlab_script << "end ;" << endl ;
			output_matlab_script << "if ( plot_sum ) ;" << endl ;
			output_matlab_script << "\tplot ( threshold , area_lt + area_gt , 'k' ) ;" << endl ;
			output_matlab_script << "\tcurve_names ( length ( curve_names ) + 1 ) = ' sum of the two ' ;" << endl ;
			output_matlab_script << "end ;" << endl ;
			output_matlab_script << "xlabel ( 'THREASHOLD ON RECTANGLE " << argv [ 2 ]  << " ') ;" << endl ;
			output_matlab_script << "ylabel ( 'RECONSTRUCTED SHAPE AREA' ) ;" << endl ;
			output_matlab_script << "title ( 'GRANULOMETRIES OF " << argv [ 1 ] << "' ) ;" << endl ;
			// output_matlab_script << "set ( gca , 'xdir' , 'reverse' ) ;" << endl ;
			output_matlab_script << "axis auto ;" << endl ;
			output_matlab_script << "legend ( curve_names , 'location' , 'south' ) ;" << endl ;
			output_matlab_script << "if ( plot_sum ) ;" << endl ;
			output_matlab_script << "\tplot ( [ " << threshold_values [ 0 ] << " , " << threshold_values [ num_rectangles - 1 ] << " ] , [ " << reconstructed_areas_gt [ 0 ] << " , " << reconstructed_areas_gt [ 0 ] << " ] , 'k' ) ;" << endl ;
			output_matlab_script << "end ;" << endl ;
			output_matlab_script << endl ;
			output_matlab_script.close () ;

			delete [] threshold_values ;
			delete [] reconstructed_areas_lt ;
			delete [] reconstructed_areas_gt ;

		}
		else {
			cerr << "WARNING : there is no rectangle !" << endl ;
		}

		cout << "Done." << endl ;

		delete measure_ptr ;
		delete volume ;

	}

	exit ( EXIT_SUCCESS ) ;

}
