/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009, 2010, 2011 Sébastien PIERARD
 */

#include "DefaultVolume2D.h"
#include <stdexcept>

DefaultVolume2D :: DefaultVolume2D ( int minX , int maxX , int minY , int maxY )
: minX ( minX ) , maxX ( maxX ) , minY ( minY ) , maxY ( maxY ) {
	if ( maxX < minX ) throw std::invalid_argument ( "maxX < minX" ) ;
	if ( maxY < minY ) throw std::invalid_argument ( "maxY < minY" ) ;
	mat = new Matrix < bool > ( minX , maxX , minY , maxY ) ;
	for ( int x = minX ; x <= maxX ; ++ x ) {
		for ( int y = minY ; y <= maxY ; ++ y ) {
			( * mat ) ( x , y ) = false ;
		}
	}
	area = 0 ;
}

bool DefaultVolume2D :: isPointInside ( int x , int y ) const {
	if ( x < minX || x > maxX ) return false ;
	if ( y < minY || y > maxY ) return false ;
	return ( * mat ) ( x , y ) ;
}

AxisAlignedBoundingBox DefaultVolume2D :: getBoundingBox () const {
	AxisAlignedBoundingBox aabb ;
	for ( int x = minX ; x <= maxX ; ++ x ) {
		for ( int y = minY ; y <= maxY ; ++ y ) {
			if ( isPointInside ( x , y ) ) {
				aabb.extendToIncludePoint ( x , y ) ;
			}
		}
	}
	return aabb ;
}

void DefaultVolume2D :: add ( const Volume2D & other ) {
	AxisAlignedBoundingBox other_aabb = other.getBoundingBox () ;
	const int other_min_x = other_aabb.getMinX () ;
	const int other_max_x = other_aabb.getMaxX () ;
	const int other_min_y = other_aabb.getMinY () ;
	const int other_max_y = other_aabb.getMaxY () ;
	if ( other_min_x < minX ) throw std::out_of_range ( "The bounding box of 'other' is not included in this's bounding box." ) ;
	if ( other_max_x > maxX ) throw std::out_of_range ( "The bounding box of 'other' is not included in this's bounding box." ) ;
	if ( other_min_y < minY ) throw std::out_of_range ( "The bounding box of 'other' is not included in this's bounding box." ) ;
	if ( other_max_y > maxY ) throw std::out_of_range ( "The bounding box of 'other' is not included in this's bounding box." ) ;
	for ( int x = other_min_x ; x <= other_max_x ; ++ x ) {
		for ( int y = other_min_y ; y <= other_max_y ; ++ y ) {
			if ( other.isPointInside ( x , y ) ) {
				if ( ( * mat ) ( x , y ) == false ) {
					++ area ;
					( * mat ) ( x , y ) = true ;
				}
			}
		}
	}
}

int DefaultVolume2D :: getArea () const {
	return area ;
}

DefaultVolume2D :: ~ DefaultVolume2D () {
	delete mat ;
}

